;; -*- Lisp -*-

;;; Rule package for Trivial Rule Engine -- Version A

;; Copyright (c) 1986, 1987, 1988, 1989, 1990, 1991, Kenneth D. Forbus,
;;  Northwestern University, and Johan de Kleer, the Xerox Corporation
;; All rights reserved.

(in-package 'user)

(proclaim '(special *atre* ;; The default ATRE, see AINTER.LISP
		    *end-forms* *rule-indexing* *bound-vars* ;; See below
		    *in-nodes* *imp-nodes*))

;;; A rule is turned into two procedures, one for matching and the
;;; other for executing the body.

;;; These variables are used during rule construction.
(defvar *bound-vars* nil)
;; *BOUND-VARS* is used internally to keep track of lexical envirionment
(defvar *end-forms* nil)
 ;; *END-FORMS* caches procedure definitions created while building rules
(defvar *rule-indexing* nil)
 ;; *RULE-INDEXING* caches forms which index rules, used while rule-building.

(defvar *in-nodes* nil)
(defvar *imp-nodes* nil)
 ;; Caches for nodes corresponding to previous triggers.

;; The next globals help ensure unique names for the procedures
;; associated with rules.  Users who write complex systems should
;; call (Rule-File <prefix string>) at the beginning of a file of rules. 
(defvar *file-counter* 0)
(defvar *file-prefix* "")

(defmacro Rule-File (prefix)
  `(eval-when (compile load eval)
     (setq *file-counter* 0)
     (setq *file-prefix* ,prefix)))

;;;; Defining rules
;;; <condition> =  :INTERN | :IN | :IMPLIED-BY
;;; If you want to mix trigger types, use nested calls to RULE.
  ;; Trigger syntax is 
  ;; (<pattern1> . <options for pattern1>  <pattern2> <options for pattern2> ...)
  ;;  and <options> can be empty, or 
  ;;  :TEST <code> and/or :VAR <var>, where <code> must be
  ;;  non-nil for the match to succeed, and <var> will be
  ;;  bound to the whole pattern.
  ;; e.g., ((Queen ?x1 ?y1) :var ?f1 (Queen ?x1 ?y2) :var ?f2 :test (not (= ?y1 ?y2)))

(defmacro rule (condition trigger-list &rest body)
  (do-rule condition (parse-triggers trigger-list) body))

(defun parse-triggers (trigger-list)
  (cond ((null trigger-list) nil)
	(t (multiple-value-bind (var test new-triggers)
				(parse-trigger-options (cdr trigger-list) nil nil)
				(cons (list (car trigger-list) var test)
				      (parse-triggers new-triggers))))))

(defun parse-trigger-options (triggers var test)
  (case (car triggers)  ;; Strips off options, returns rest of triggers
	(:var (parse-trigger-options (cddr triggers) (cadr triggers) test))
	(:test (parse-trigger-options (cddr triggers) var (cadr triggers)))
	(t (values var test triggers))))

;;;; Orchestrating the rule expansion

(defun do-rule (condition triggers body)
  (let ((*end-forms* nil) ;; accumulates procedure definitions
	(*rule-indexing* nil)) ;; accumulates indexing forms
    ;; Uncomment these if you have a serious rule expansion problem to track down.
    ;;(format t "~% Adding rule with triggers: ") (pprint triggers)
    ;;(format t "~% And body: ") (pprint body)
  (add-rule condition (car triggers)
	    (subst 'internal-rule 'rule
		   (make-nested-rule condition 
		    (cdr triggers) body)))
  ;; Returning this ensures that all procedure definitions
  ;; are executed before any indexing occurs.
  `(progn ,@ *end-forms* ,@ *rule-indexing*)))

(defmacro internal-rule (condition triggers-in &rest body)
  ;; All but the rule corresponding to the outermost
  ;; trigger are internal rules.
  (let ((triggers (parse-triggers triggers-in)))
    `(add-internal-rule ,condition
			,(car triggers)
			,(make-nested-rule condition (cdr triggers) body))))

(defun make-nested-rule (condition triggers body)
  (cond ((null triggers) body)
	(t `((add-internal-rule ,condition
	       ,(car triggers)
	       ,(make-nested-rule condition (cdr triggers) body))))))

(defun add-rule (condition trigger body)
  ;; Must be executed after functions are defined.
  (push (build-rule condition trigger body) *rule-indexing*)
  nil)

(defmacro add-internal-rule (condition trigger body)
  ;; The form to index this rule must appear in
  ;; the body of the rule which directly contains it.
  (build-rule condition trigger body))

;;;; Building rules

(defun build-rule (condition trigger body &aux match-procedure body-procedure)
  (let ((pattern (car trigger))
	(var (cadr trigger))
	(test (caddr trigger)))
   (setq match-procedure (generate-match-procedure pattern var test condition))   
   (setq body-procedure (generate-body-procedure pattern condition var body))
   (push match-procedure *end-forms*)
   (push body-procedure *end-forms*)
   `(insert-rule
     (get-class ,(get-trigger-class pattern))
     ;return form to index rule
       ,(if *bound-vars* ;the match function for rule
	    `(function (lambda (p)
	       (,(cadr match-procedure) p ,@ *bound-vars*)))
	  `(function ,(cadr match-procedure)))
     (function ;;the body function for rule
       ,(if (or *bound-vars*
		(not (eq condition ':INTERN)))
	    (let ((tv (nreverse
			(pattern-free-variables trigger))))
	      (unless (eq condition ':INTERN) (push 'TRIGGER-NODE tv))
	      `(lambda ,tv
		 (,(cadr body-procedure) ,@ tv
		  ;(fn-name parameters)
		  ,@ (scratchout tv *bound-vars*))))
	      (cadr body-procedure)))
     *in-nodes*       ;; TMS nodes inherited from the external rule contours
     *imp-nodes*))) 

(defun get-trigger-class (trigger)
  (cond ((null trigger) (error "Null trigger in ATRE rule"))
	((variable? trigger)
	 (if (member trigger *bound-vars*)  trigger
	     (error "~%Trigger class is unbound -- ~A."
		    trigger)))
	((symbolp trigger)  (list 'QUOTE trigger))
	((listp trigger) (get-trigger-class (car trigger)))
	(t (error "ATRE rule trigger must be symbol or list: ~A" trigger))))

;;;; Generating the body procedure

(defmacro with-pushed-variable-bindings (new-bindings
					  &rest body)
;; generate-body-procedure needs this
  `(let ((*bound-vars* (append ,new-bindings
			       (scratchout ,new-bindings
					   *bound-vars*))))
     ,@ body))

(defun generate-body-procedure (pattern condition var body
				    &aux newly-bound env fname)
  (setq newly-bound (pattern-free-variables pattern))
  (if var (push var newly-bound))
  (setq body (with-pushed-variable-bindings
	       newly-bound (fully-expand-body body)))
  (setq env (append newly-bound
		    (scratchout newly-bound *bound-vars*)))
  (unless (eq condition ':INTERN) (push 'trigger-node env))
  (setq fname (generate-rule-procedure-name pattern))
  `(defun ,fname ,env
     ,@ (cond ((eq condition ':INTERN) body) ;; Just do it
	      (t ;; Must check and see if the node's belief state
	         ;; matches the rule's requirements
	       `((cond ((,(case condition
				(:IN 'tms-node-label) ;; Non-empty label
				(:IMPLIED-BY 'tms-node-label) ;; Same, filter later
				(t (error "~A bad condition -- GBF"
						 condition)))
			 TRIGGER-NODE) ,@ body)
		       (t (push (list ',fname ,@ env)
				(tms-node-rules TRIGGER-NODE)))))))))

(defun scratchout (l1 l2)
  ;non-destructive and order-preserving
  (dolist (el1 l1 l2) (setq l2 (remove el1 l2))))

(defun generate-rule-procedure-name (pattern)
  (intern (format nil "~A-~A-~A" 
		  *file-prefix* pattern (incf *file-counter*))))

;;;; Recursive macroexpansion
;;   Knowing what forms to skip over is still implementation
;;   dependent.  It gets hairier when one allows assertions to
;;   have procedural side-effects.

(defun fully-expand-body (body)
  (cond ((null body) nil)
	((not (listp body)) body)
	((symbolp (car body))
	 (case (car body)
	   (LAMBDA `(lambda ,(cadr body)
		      ,@ (fully-expand-body (cddr body))))
	   (DO `(do ,(cadr body) ,(caddr body)
		  ,@ (cdddr body)))
	   #+Lucid
	   (BQ-LIST* `(BQ-LIST* ,@ (mapcar #'fully-expand-body (cdr body))))
	   ((DOTIMES DOLIST LET PROG PROGN PROGV)
	    `(,(car body) ,(cadr body)
	      ,@ (fully-expand-body (cddr body))))
	   (AND (cond ((cdr body) `(AND ,@ (fully-expand-body (cdr body))))
		      (t body)))
	   (OR (cond ((cdr body) `(OR ,@ (fully-expand-body (cdr body))))
		     (t body)))
	   (MULTIPLE-VALUE-BIND
 `(multiple-value-bind ,(cadr body) ,(fully-expand-body (caddr body)) 
    ,@ (fully-expand-body (cdddr body))))	   
	   (t (let ((new-body (macroexpand body)))
		(cons (fully-expand-body (car new-body))
		      (fully-expand-body (cdr new-body)))))))
	(t (let ((new-body (macroexpand body)))
	     (cons (fully-expand-body (car new-body))
		   (fully-expand-body (cdr new-body)))))))

;;;; Running rules

(defun insert-rule (class matcher body in-nodes imp-nodes &aux rule atre)
  (setq atre (class-atre class))
  (setq rule (make-rule :matcher matcher
			:atre atre
			:body body
			:class class
			:counter (incf (atre-rule-counter atre))
			:in-nodes in-nodes
			:imp-nodes imp-nodes))
  ;; Index it
  (push rule (atre-rules atre))
  (push rule (class-rules class))
  (dolist (candidate (class-facts class))
	  (try-rule-on rule candidate)))

(defun try-rules (datum)
  (dolist (rule (class-rules (datum-class datum)))
    (try-rule-on rule datum)))

(defun try-rule-on (rule datum &aux a)
  (setq a (datum-atre datum))
  (multiple-value-bind (okay? bindings condition)
      (funcall (rule-matcher rule) (datum-lisp-form datum))
    (when okay?
      (when (or (eq condition ':IN)
		(eq condition ':IMPLIED-BY))
	(push (datum-tms-node datum) bindings))
      ;; More hard-core debugging..
;      (format t "~%    ...Queueing rule ~A on ~A and nodes ~A"
;	      rule bindings (rule-nodes rule))
      (enqueue (list (rule-body rule) bindings
		     (case condition
		       (:IN (cons (cons (datum-tms-node datum)
					(rule-in-nodes rule))
				  (rule-imp-nodes rule)))
		       (:IMPLIED-BY
			(cons (rule-in-nodes rule)
			      (cons (datum-tms-node datum)
				    (rule-imp-nodes rule))))
		       (:INTERN (cons (rule-in-nodes rule)
				      (rule-imp-nodes rule)))))
	        a))))

(defun run-rules (&optional (*atre* *atre*))
  (check-contradiction-rules *atre*) ;; Do this first 
  ;; This next part is dreadful.  Must really hook more deeply into the ATMS
  ;; to do better.  But GC'ing nodes seems ugly, too...
  (setf (atre-queue *atre*) (nconc (atre-queue *atre*) (atre-in-rules *atre*)))
  (setf (atre-in-rules *atre*) nil)
  (do ((form (dequeue *atre*) (dequeue *atre*))
       (counter 0 (1+ counter)))
      ((null form)
       (debugging-atre "~%    ~A rules run."  counter)
       (values counter (incf (atre-rules-run *atre*) counter)))
      (execute-rule form *atre*)))

;;;; Executing rules, checking for appropriate conditions

(defun execute-rule (queued-rule atre)
  ;; Now is (<procedure> <arguments> <node list>)
  ;; Check the node list before executing, to make sure
  ;; all the belief conditions are satisifed.
  (let ((*in-nodes* (car (third queued-rule)))
	(*imp-nodes* (cdr (third queued-rule))))
    (unless (in-triggers-ready? *in-nodes* atre)
      ;; Re-queue under ATRE for checking.
      ;; ****** Introduce temporary nodes?  Cache rules
      ;; ****** on justifications?
      (push queued-rule (atre-in-rules atre))
      (return-from EXECUTE-RULE nil))
    (unless (implied-by-triggers-ready? *imp-nodes* atre)
      (push queued-rule (atre-imp-rules atre))
      (return-from EXECUTE-RULE nil))
    ;; Let's do it
    (apply (car queued-rule) (cadr queued-rule))))

(defun in-triggers-ready? (nodes atre &optional (env (atms-empty-env
						      (atre-atms atre))))
  (cond ((env-nogood? env) nil) ;; Combination was nogood
	((null nodes) t) ;; Nothing else to combine
	(t (dolist (new (tms-node-label (car nodes)))
	     (let ((u (union-env new env)))
	       (if (in-triggers-ready? (cdr nodes) atre u)
		   (return-from IN-TRIGGERS-READY? t)))))))

(defun implied-by-triggers-ready? (nodes atre)
  (or (null nodes) ;; No triggers, no problem
      (and (focus-okay? atre)
	   (every #'(lambda (n) 
		      (in-node? n (atre-focus atre)))
		  nodes))))

(defun rules-waiting? (atre) (atre-queue atre))

(defun enqueue (new a) (push new (atre-queue a)))

(defun dequeue (atre)
  (if (atre-queue atre) (pop (atre-queue atre))))

;;;; Display routines

(defun show-rules (&optional (atre *atre*) (stream *standard-output*)
			&aux counter dist inc contra imp in
			queued)
  (setq counter 0)
  (dolist (class (atre-classes atre))
    (setq inc (length (class-rules class)))
    (when (> inc 0)
      (push (cons (class-name class) inc) dist)
      (incf counter inc)))
  (setq in (length (atre-in-rules atre))
	imp (length (atre-imp-rules atre))
	contra 0 queued (length (atre-queue atre)))
  (dolist (entry (atre-contradiction-rules atre))
    (incf contra (length (cdr entry))))
  (setq counter (+ in imp contra counter))
  (format stream "~% ~A has ~D rules in all." (atre-title atre) counter)
  (format stream "~%  ~A queued." (if (> queued 0) queued "None"))
  (if (> (+ in imp contra) 0)
      (format stream "  Pending: ~A in, ~A implied-by, ~A contradiction."
	      (if (> in 0) in "No") (if (> imp 0) imp "No") (if (> contra 0) contra "No"))
      (format stream "  None pending."))
  (when dist
    (format stream "~% Cached under classes:")
    (dolist (entry dist)
      (format stream "~%    ~A: ~D" (car entry) (cdr entry))))
  atre)

(defun print-rules (&optional (atre *atre*) (stream *standard-output*) &aux counter)
  (setq counter 0)
  (format t "~%The rules in ~A are:" (atre-title atre))
  (dolist (rule (atre-rules atre))
	  (incf counter)
	  (print-rule rule stream))
    counter)

(defun print-rule (rule &optional (stream *standard-output*))
  (format stream "~% ~A: ~A, ~A"
	  rule (rule-matcher rule) (rule-body rule)))

(defun test-rule-expansion ()
 (pprint (macroexpand
	  '(rule :IN  ((implies ?p ?q) :var ?f1 ?p)
		 (rassert! ?q (CE ?f1 ?p))))))

(defun get-rule (num &optional (atre *atre*))
  (dolist (rule (atre-rules atre))
    (when (= (rule-counter rule) num) (return-from GET-RULE rule))))
