;; -*- Lisp -*-

;;; Natural deduction rules for JTRE

;; Copyright (c) 1988, Kenneth D. Forbus, University of Illinois
;; All rights reserved.

;; These rules are based on a natural deduction system
;; developed by Kalish and Montigue, which organizes rules
;; in terms of introducing and eliminating connectives.
;; This provides a natural organization for our inference
;; rules.

;; As in the written rules, the predicate "SHOW" will indicate
;; our interest in a proving a fact of a particular form.

(defvar *debug-nd* nil)

(defmacro debug-nd (msg &rest args)
  `(when *debug-nd*
	 (format t ,msg ,@ args)))

;; We begin by Modus ponens (conditional elimination)
;; and its friends.

(rule *jtre* ((:in (implies ?p ?q) :var ?f1)
	      (:in ?p)) ;; conditional elimination
      (debug-nd "~%~D: CE: ~A" (jtre-depth *jtre*) ?q)
      (rassert! ?q (CE ?f1 ?p)))

(rule *jtre* ((:in (show ?q) :var ?f1)) ;; backward chaining.
      (rule *jtre* ((:in (implies ?p ?q) :var ?f2))
	    (rassert! (show ?p) (BC-CE ?f1 ?f2))))

(a-rule *jtre* ((:in (show (implies ?p ?q)) :var ?f1)) ;; Conditional Introduction
 (unless (already-assumed? `(implies ,?p ,?q) *jtre*)
   (debug-nd "~%~D: Trying CI on (implies ~A ~A)."
	     (jtre-depth *jtre*) ?p ?q)
   (multiple-value-bind (win? asns)
     (seek-in-context ?p ?q *jtre*)
     (debug-nd "~%~D: CI: (implies ~A ~A) ~A." (jtre-depth *jtre*) ?p ?q
	       (if win? "succeeded" "failed"))
     (when win?
	   (assert! `(implies ,?p ,?q) *jtre* `(CI ,@ asns))))))

;;;; And elimination and introduction

(rule *jtre* ((:in (and . ?conjuncts) :var ?f1)) ;AND elimination
      (dolist (conjunct ?conjuncts)
	(debug-nd "~%~D: AE: ~A" (jtre-depth *jtre*) conjunct)
	(assert! conjunct *jtre* `(AE ,?f1))))

(rule *jtre* ((:IN (show (and ?c1 ?c2)) :var ?f1))
      (rassert! (show ?c1) (BC-AI ?f1)) ;; Work in parallel
      (rassert! (show ?c2) (BC-AI ?f1)) ;; Record interest dependencies properly   
      ;; Notice that validity does not depend on interest. 
      (rule *jtre* ((:in ?c1) (:in ?c2)) (rassert! (and ?c1 ?c2) (AI ?c1 ?c2))))

(rule *jtre* ((:IN (show (and ?c1 ?c2 ?c3)) :var ?f1))
      (rassert! (show ?c1) (BC-AI ?f1))
      ;; For variety, this one is done serially
      (rule *jtre* ((:IN ?c1))
	    (rassert! (show ?c2) (BC-AI ?f1 ?c1))
	    (rule *jtre* ((:in ?c2))
		  (rassert! (show ?c3) (BC-AI ?f1 ?c1 ?c2))
		  (rule *jtre* ((:in ?c3)) (rassert! (and ?c1 ?c2 ?c3) (AI ?c1 ?c2 ?c3))))))

;;;; Biconditional elimination and introduction

(rule *jtre* ((:in (iff ?p ?q) :var ?f1)) ;; IFF elimination
      (debug-nd "~%~D: BE: ~A~%~D: BE: ~A"
		(jtre-depth *jtre*) `(implies ,?p ,?q) (jtre-depth *jtre*)
		`(implies ,?q ,?p))
      (rassert! (implies ?p ?q) (BE ?f1))
      (rassert! (implies ?q ?p) (BE ?f1)))

(rule *jtre* ((:in (show (iff ?p ?q)) :var ?f1)) ;IFF introduction
      (debug-nd "~%~D: BC-BI: (show (implies ~A ~A))"
		(jtre-depth *jtre*) ?p ?q)
      (debug-nd "~%~D: BC-BI: (show (implies ~A ~A))" 
		(jtre-depth *jtre*) ?q ?p)
      (rassert! (show (implies ?p ?q)) (BC-BI ?f1))
      (rassert! (show (implies ?q ?p)) (BC-BI ?f1))
      (rule *jtre* ((:in (implies ?p ?q) :var ?f2)
		    (:in (implies ?q ?p) :var ?f3))
	    (debug-nd "~%~D: BI: ~A"
		      (jtre-depth *jtre*) `(iff ,?p ,?q))
	    (rassert! (iff ?p ?q) (BI ?f2 ?f3))))

;;;; Dealing with negation

(rule *jtre* ((:in (not (not ?p)) :var ?f1))
      (debug-nd "~%~D: NE: ~A" (jtre-depth *jtre*) ?p)
      (rassert! ?p (NE ?f1))) ;; NOT elimination

(a-rule *jtre* ((:in (show (not ?p)) :var ?f1)) ;; NOT introduction
	(unless (eq ?p 'contradiction) ;; This one wouldn't make sense
	  (debug-nd "~%~D: NI attempt: (not ~A)" (jtre-depth *jtre*) ?p)
	  (multiple-value-bind (found? asns)
			       (seek-in-context ?p nil *jtre*)
	    (debug-nd "~%~D: NI: ~A" (jtre-depth *jtre*) `(not ,?p))
	    (assert! `(not ,?p) *jtre* `(NI ,@ asns)))))

;;;; Disjunction elimination and introduction

(rule *jtre* ((:in (show (or . ?disjuncts)) :var ?f1)) ;; OR introduction
      (dolist (?disjunct ?disjuncts)
	      (debug-nd "~%~D: OI-BC: (show ~A)"
			(jtre-depth *jtre*)  ?disjunct)
       (rlet ((?disjunct ?disjunct))
	     (rassert! (show ?disjunct) (BC-OI ?f1))
	     (rule *jtre* ((:in ?disjunct)) 
		   (debug-nd "~%~D: OI: ~A"
			     (jtre-depth *jtre*) (cons 'OR ?disjuncts))
		   (assert! `(or ,@ ?disjuncts) *jtre* `(OI ,?disjunct))))))

(rule *jtre* ((:in (show ?r) :var ?f1)) ;; OR elimination
  (unless (or (eq ?r 'contradiction)
	      (not (simple-proposition? ?r))
	      (already-assumed? ?r *jtre*))
    ;; Turns out to be hairy to do the (eval (make-nested-rule...)) part
    ;; So return to simple two-disjunct version.
    (rule *jtre* ((:in (or ?d1 ?d2) :var ?f2))
	  (unless (or (eq ?d1 'contradiction)
		      (eq ?d2 'contradiction))
	    (debug-nd "~%~D: OE-BC: (show (implies ~A ~A))"
		      (jtre-depth *jtre*) ?d1 ?r)
	    (rassert! (show (implies ?d1 ?r)) (BC-OE ?f1 ?f2))	 
	    (debug-nd "~%~D: OE-BC: (show (implies ~A ~A))"
		      (jtre-depth *jtre*) ?d2 ?r)
	    (rassert! (show (implies ?d2 ?r)) (BC-OE ?f1 ?f2))
	    (rule *jtre* ((:in (implies ?d1 ?r) :var ?f3)
			  (:in (implies ?d2 ?r) :var ?f4))
	     (debug-nd "~% ~D: OE: ~A" (jtre-depth *jtre*) ?r)
	     (rassert! ?r (OE ?f2 ?f3 ?f4)))))))

;;;; Indirect proof and contradiction detection

(a-rule *jtre* ((:in (show ?p) :var ?f1)) ;indirect proof.
  (unless (or (eq ?p 'contradiction)
	      (not (simple-proposition? ?p)))
    (debug-nd "~%~D: IP attempt: ~A."
	      (jtre-depth *jtre*) ?p)
    (multiple-value-bind (found? asns)
	(seek-in-context `(not ,?p) nil *jtre*)
	(debug-nd "~%~D: IP: ~A ~A" (jtre-depth *jtre*) ?p
		  (if found? "succeeded." "failed."))
      (when found? (assert! ?p *jtre* `(IP ,@ asns))))))

(rule *jtre* ((:in (not ?p))
	      (:in ?p))
      ;; Assume contradiction detection is always interesting
	(debug-nd "~%~D: Contra: (not ~A) and ~A"
		  (jtre-depth *jtre*) ?p ?p)
	(rassert! contradiction (Lossage ?p (not ?p))))
