;-*- Mode:  LISP; Syntax: Common-lisp -*-

;;;; Example of dependency-directed search using TRE-J
;;
;; Copyright (C) 1986, 1990 Kenneth D. Forbus, All rights reserved.
;;
;; This program uses dependency-directed search to solve the n-queens
;; problem.  The input is the size of the chessboard.  We follow the usual
;; trick and assume that exactly one queen will be placed in each column.
;; Then the problem reduces to finding a set of assignments for the
;; rows such that the queens do not capture each other.

(defvar *queen-assertions* nil)
(defvar *queen-solutions* nil)
(defvar *n-assumptions* 0)
(defvar *n-tests* 0)
(defvar *queens-debug* t)

(defun n-queens (n &optional (debugging? nil))
  (setup-queens-problem n debugging?)
  (solve-queens-problem *queen-assertions*)
  (length *queen-solutions*))

(defun queens-contradiction-handler (contradictions ignore)
  (if *queens-debug*
      (format t "~%    Contradiction: ~A, ~A" 
	      (car contradictions)
	      (mapcar #'view-node (assumptions-of-clause
				   (car contradictions)))))
  (throw ':QUEENS-LOST
	 (cons :LOSERS (assumptions-of-clause (car contradictions)))))

(defvar *queens-rules* "/u/bps/code/ltms/lqrule.lisp")

(defun setup-queens-problem (n debugging?)
  (setq *LTRE* (create-LTRE (format nil "~D Queens LTRE" n)
			    :debugging debugging?))
  (in-LTRE *LTRE*)
  (setq *queen-solutions* nil)
  (setq *n-assumptions* 0 *n-tests* 0)
  (change-ltms (ltre-ltms *ltre*)
	       :contradiction-handler 'queens-contradiction-handler)
  (load *queens-rules*)
  (install-queen-assertions n))

(defun install-queen-assertions (n)
  (do ((column 1 (1+ column))
       (queen nil)
       (column-queens nil nil)
       (assertions nil))
      ((> column n) (setq *queen-assertions* assertions))
    (dotimes (row n)
      (setq queen `(Queen ,column ,(1+ row)))
      (referent queen *LTRE* T) ;;Install it in the database
      (push queen column-queens))
    (push column-queens assertions)))

(defun solve-queens-problem (asserts)
  (cond ((null asserts) (gather-queens-solution))
	(t (dolist (this-queen (car asserts))
	     (unless (false? this-queen)
	       ;respect nogood information
	       (when *queens-debug*
		 (format t "~% Trying QUEEN(~D,~D).."
			 (cadr this-queen) (caddr this-queen)))
	       (let ((result
		       (catch ':QUEENS-LOST
			 (progn (assume! this-queen 'queen-assumption)
				(run-rules)))))
		 (incf *n-assumptions*)
		 (cond ((and (listp result)
			     (eq (car result) ':losers))
			;;This assumption lost, so justify the negation
			;; based on the other relevant assumptions.
			(retract! this-queen 'queen-assumption)
			(when *queens-debug*
			  (format t "~% Lost, retracting QUEEN(~D,~D).."
				  (cadr this-queen) (caddr this-queen)))
			(add-nogood (get-tms-node this-queen)
				    :TRUE (cdr result)))
		       (t ;okay so far, go down to next level
			 (solve-queens-problem (cdr asserts))
			  (when *queens-debug*
			    (format t "~% Backtracking, retracting QUEEN(~D,~D).."
				    (cadr this-queen) (caddr this-queen)))
			 (retract! this-queen 'queen-assumption)))))))))

(defun queens-so-far ()
  (remove-if-not #'true? (fetch `(Queen ?c ?r))))

(defun gather-queens-solution ()
  (push (queens-so-far)	*queen-solutions*))

(defun test-queens (from to)
  (do ((n from (1+ n)))
      ((> n to))
    (run-time (n-queens n))
    (format t "~%~D solutions, ~D assumptions." (length *queen-solutions*)
	    *n-assumptions*)))
