;;; -*- Mode: LISP; Syntax: Common-lisp; Package: USER -*-

;;; Logic-based truth maintenance system

;;; Last edited: 6/9/91

(in-package 'user)

;;;; Copyright (c) 1986-1991,  Kenneth D. Forbus, Northwestern
;;;; University and Johan de Kleer, Xerox Corporation.
;;;; All rights reserved.

(defstruct (ltms (:PRINT-FUNCTION print-ltms))
  (title nil)
  (node-counter 0)              ; unique namer for nodes.
  (clause-counter 0)            ; unique namer for justifications.
  (nodes nil)                   ; list of all ltms nodes.
  (clauses nil)                 ; list of all clauses.
  (debugging nil)               ; debugging flag
  (checking-contradictions t)
  (node-printer nil)			       
  (contradiction-handlers nil)
  (pending-contradictions nil)
  (enqueue-procedure nil)
  (complete nil)                ; Is this a complete LTMS?
  (violated-clauses nil)        ;
  (pis nil))

(defun print-ltms (ltms stream ignore)
  (declare (ignore ignore))
  (format stream "#<LTMS: ~A>" (ltms-title ltms)))

(defstruct (tms-node (:PRINT-FUNCTION print-tms-node))
  (index 0)                    ; unique namer for nodes
  (datum nil)		       ; positive inference engine datum.
  (label :UNKNOWN)             ; :UNKNOWN, :TRUE, or :FALSE.
  (support nil)                ; clause which supports it,
  (true-clauses nil)           ; clauses in which this node is true
  (false-clauses nil)          ; clauses in which this node is false
  (mark nil)                   ; marker for sweep algorithms
  (assumption? nil)
  (true-rules nil)             ; rules run when the node is true
  (false-rules nil)            ; rules run when the node is false
  (ltms nil)                   ; LTMS it is part of.
  (pis nil))                   ; pis when last resolved.

(defun print-tms-node (node stream ignore)  (declare (ignore ignore))
  (format stream "#<NODE: ~A>" (node-string node)))

(defstruct (clause (:PRINT-FUNCTION print-clause))
  (index 0)       ; Unique namer
  (informant nil)
  (literals nil)  ; a list of (<node> . <truth>)
  (pvs 0))        ; Number of terms which potentially violate it.

(defun print-clause (clause stream ignore)   (declare (ignore ignore))
  (format stream "#<Clause ~D>" (clause-index clause)))

;;;; More basic definition stuff

(defun node-string (node)
  (funcall (ltms-node-printer (tms-node-ltms node)) node))

(defmacro debugging-ltms (ltms msg &optional node &rest args)
  `(when (ltms-debugging ,ltms)
     (format *trace-output*
	     ,msg (if ,node (node-string ,node)) (progn ,@args))))

(defun ltms-error (string &optional thing) (error string thing))

(defun default-node-printer (n)
  (format nil "~A" (tms-node-datum n)))

(defun flipt (label)
  (ecase label (:TRUE :FALSE) (:FALSE :TRUE) (:UNKNOWN :UNKNOWN)))

;;; Basic inference-engine interface.

(defun create-ltms (title &key (node-printer 'default-node-printer)
		               (debugging NIL)
			       (checking-contradictions T)
			       (contradiction-handler 'ask-user-handler)
			       (enqueue-procedure NIL)
			       (complete nil))
  (make-ltms :TITLE title :NODE-PRINTER node-printer
	     :DEBUGGING debugging
	     :CHECKING-CONTRADICTIONS checking-contradictions
	     :ENQUEUE-PROCEDURE enqueue-procedure
	     :CONTRADICTION-HANDLERS (list contradiction-handler)
	     :COMPLETE complete))

(defun change-ltms (ltms &key (contradiction-handler nil contra?)
		              node-printer
			      enqueue-procedure
			      (debugging nil debugging?)
			      (checking-contradictions nil checking?)
			      (complete nil complete?))
  (if node-printer (setf (ltms-node-printer ltms) node-printer))
  (if debugging? (setf (ltms-debugging ltms) debugging))
  (if checking? (setf (ltms-checking-contradictions ltms)
		      checking-contradictions))
  (if contra?
      (setf (ltms-contradiction-handlers ltms)
	    (list contradiction-handler)))
  (if enqueue-procedure
      (setf (ltms-enqueue-procedure ltms) enqueue-procedure))
  (if complete? (setf (ltms-complete ltms) complete)))

(defun unknown-node? (node) (eq (tms-node-label node) :UNKNOWN))

(defun known-node? (node) (not (eq (tms-node-label node) :UNKNOWN)))

(defun true-node? (node) (eq (tms-node-label node) :TRUE))

(defun false-node? (node) (eq (tms-node-label node) :FALSE))

(defun tms-create-node (ltms datum &key assumptionp)
  (let ((node (make-tms-node :INDEX (incf (ltms-node-counter ltms))
			     :DATUM datum :ASSUMPTION? assumptionp
			     :LTMS ltms)))
    (push node (ltms-nodes ltms))
    node))

(defun enable-assumption (node label)
  (cond ((not (tms-node-assumption? node))
	 (ltms-error "Can't enable the non-assumption ~A" node))
	((eq (tms-node-label node) label)
	 (setf (tms-node-support node) :ENABLED-ASSUMPTION))
	((eq (tms-node-label node) :UNKNOWN)
	 (top-set-truth node label :ENABLED-ASSUMPTION))
	(t (ltms-error "Can't set an already set node" node))))

(defun convert-to-assumption (node)
  (unless (tms-node-assumption? node)
    (debugging-ltms (tms-node-ltms node)
	       "~%Converting ~A into an assumption" node)
    (setf (tms-node-assumption? node) T)))

;;; Theorem : This can never cause contradiction handling.
(defun retract-assumption (node)
  (when (and (known-node? node)
	     (eq (tms-node-support node) :ENABLED-ASSUMPTION))
    (find-alternative-support (tms-node-ltms node)
			      (propagate-unknownness node))))
    

;;; Adding formulas to the LTMS.
(defun add-formula (formula &optional informant &aux ltms)
  (setq informant (list :IMPLIED-BY formula informant))
  (dolist (clause (conj-normal formula))
    (unless ltms (setq ltms (tms-node-ltms (caar clause))))
    (if (setq clause (simplify-clause clause))
	(add-clause-internal clause informant T)))
  (check-for-contradictions ltms))
  
(defun simplify-clause (clause)
  (do ((term-pairs clause (cdr term-pairs))
       (so-far nil)
       (matcher nil))
      ((null term-pairs) so-far)
    (setq matcher (assoc (caar term-pairs) so-far
			 :TEST #'equal))
    (cond ((null matcher) (push (car term-pairs) so-far)) ;okay
	  ((eq (cdar term-pairs) (cdr matcher))) ;ignore redundant clause
	  (t (return nil)))))

(defun conj-normal (exp)
  (cond ((null exp) nil)
	((not (listp exp)) `(((,exp . :TRUE))))
	(t (case (car exp)			
	     (:IMPLIES (disjunct (neg-conj-normal (cadr exp))
				 (conj-normal (caddr exp))))
	     (:OR (do ((result (conj-normal (cadr exp)))
		       (rest (cddr exp) (cdr rest)))
		      ((null rest) result)
		    (setq result (disjunct (conj-normal (car rest)) result))))
	     (:AND (mapcan #'conj-normal (cdr exp)))
	     (:NOT (neg-conj-normal (cadr exp)))
	     (:IFF (nconc (conj-normal `(:implies ,(cadr exp) ,(caddr exp)))
			  (conj-normal `(:implies ,(caddr exp) ,(cadr exp)))))
	     (:TAXONOMY
	       (conj-normal
		 `(:and (:or ,@ (cdr exp)) ;one must be true
			,@ (do ((firsts (cdr exp) (cdr firsts))
				(rests (cddr exp) (cdr rests))
				(result nil))
			       ((null rests) result)
			       (dolist (other rests)
				       (push `(:not (:and ,(car firsts)
							  ,other)) result))))))
	     (t `(((,exp . :TRUE))))))))

;;; Clause translation, continued

(defun neg-conj-normal (exp)
  (cond ((null exp) nil)
	((not (listp exp)) `(((,exp . :FALSE))))
	(t (case (car exp)
	     (:IMPLIES (nconc (conj-normal (cadr exp))
			      (neg-conj-normal (caddr exp))))
	     (:AND (do ((result (neg-conj-normal (cadr exp)))
			(rest (cddr exp) (cdr rest)))
		       ((null rest) result)
		     (setq result
			   (disjunct (neg-conj-normal (car rest)) result))))
	     (:OR (mapcan #'neg-conj-normal (cdr exp)))
	     (:NOT (conj-normal (cadr exp)))
	     (:IFF (nconc (conj-normal `(:implies ,(cadr exp) ,(caddr exp)))
			  (conj-normal `(:implies ,(caddr exp) ,(cadr exp)))))
	     (t `(((,exp . :FALSE))))))))

(defun disjunct (conj1 conj2)
  (mapcan #'(lambda (disj1)
	    (mapcar #'(lambda (disj2) (append disj1 disj2))
		    conj2))
	  conj1))

;;; Adding clauses
(defun add-clause (true-nodes false-nodes &optional informant)
  (add-clause-internal (nconc (mapcar
				#'(lambda (tn) (cons tn :TRUE))
				true-nodes)
			      (mapcar
				#'(lambda (fn) (cons fn :FALSE))
				false-nodes))
		       informant
		       nil))

(defun add-clause-internal (literals informant internal &aux ltms)
  (unless literals
	  (ltms-error "Total contradiction: Null clause" informant))
  (setq ltms (tms-node-ltms (caar literals)))
  (if (ltms-complete ltms)
      (full-add-clause ltms (list (cons literals informant)))
      (bcp-add-clause ltms literals informant))
  (unless internal (check-for-contradictions ltms)))

(defun bcp-add-clause (ltms literals informant &aux cl)
  (setq cl (make-clause :INDEX (incf (ltms-clause-counter ltms))
			:LITERALS literals
			:INFORMANT informant))
  (push cl (ltms-clauses ltms))
  (dolist (term-pair (clause-literals cl))
    (unless (eq (tms-node-label (car term-pair))
		(flipt (cdr term-pair)))
      (incf (clause-pvs cl))))
  (dolist (term literals)
    (ecase (cdr term)
      (:TRUE (push cl (tms-node-true-clauses (car term))))
      (:FALSE (push cl (tms-node-false-clauses (car term))))))
  (check-clauses ltms (list cl))
  cl)

(defun add-nogood (culprit sign assumptions &aux trues falses)
  (dolist (a assumptions (add-clause trues falses 'NOGOOD))
    (ecase (if (eq a culprit) sign (tms-node-label a))
      (:TRUE (push a falses))
      (:FALSE (push a trues)))))

;;; Boolean Constraint Propagation.

(proclaim '(special *clauses-to-check*))

(defun check-clauses (ltms *clauses-to-check*)
  (debugging-ltms ltms "~% Beginning propagation...")
  (do nil ((null *clauses-to-check*))
    (check-clause ltms (pop *clauses-to-check*))))

(defun check-clause (ltms clause &aux unknown-pair)
  (cond ((violated-clause? clause)
	 (push clause (ltms-violated-clauses ltms)))
	((= (clause-pvs clause) 1)
	 ;; Exactly one term of the clause remains that can
	 ;; satisfy the clause, so deduce that term
	 (setq unknown-pair (find-unknown-pair clause))
	 (when unknown-pair ;must check, because it might have other
	   (set-truth (car unknown-pair) ; support
		      (cdr unknown-pair) clause)))))

(defun violated-clause? (clause) (= (clause-pvs clause) 0))

(defun find-unknown-pair (clause)
  (dolist (term-pair (clause-literals clause))
    (if (unknown-node? (car term-pair)) (return term-pair))))

(defun top-set-truth (node value reason &aux *clauses-to-check*)
  (set-truth node value reason)
  (check-clauses (tms-node-ltms node) *clauses-to-check*)
  (check-for-contradictions (tms-node-ltms node)))

(defun set-truth (node value reason &aux ltms enqueuef)
  (setq ltms (tms-node-ltms node)
	enqueuef (ltms-enqueue-procedure ltms))
  (debugging-ltms ltms "~%  Setting ~A to ~A, via ~A." node value reason)
  (setf (tms-node-support node) reason)
  (setf (tms-node-label node) value)
  (ecase value ;figure out which set of rules to queue up
    (:TRUE (when enqueuef
	     (dolist (rule (tms-node-true-rules node))
	       (funcall enqueuef rule))
	     (setf (tms-node-true-rules node) nil))
	   (dolist (clause (tms-node-false-clauses node))
	     (if (< (decf (clause-pvs clause)) 2)
		 (push clause *clauses-to-check*))))
    (:FALSE (when enqueuef
	      (dolist (rule (tms-node-false-rules node))
		(funcall enqueuef rule)))
	    (setf (tms-node-false-rules node) nil)
	    (dolist (clause (tms-node-true-clauses node))
	      (if (< (decf (clause-pvs clause)) 2)
		  (push clause *clauses-to-check*))))))

;;; Retracting an assumption.
(defun propagate-unknownness (in-node)
  (let (node old-value node2 unknown-queue)
    (do ((forget-queue (cons in-node nil) (nconc forget-queue new))
	 (new nil nil))
	((null forget-queue) unknown-queue)
      (setq forget-queue (prog1 (cdr forget-queue)
				(rplacd forget-queue unknown-queue)
				(setq unknown-queue forget-queue))
	    node (car unknown-queue))
      (debugging-ltms (tms-node-ltms in-node) "~% Retracting ~A." node)
      (setq old-value (tms-node-label node))
      (setf (tms-node-label node) :UNKNOWN)
      (setf (tms-node-support node) nil)
      (ecase old-value
	(:TRUE (dolist (clause (tms-node-false-clauses node))
		 (when (= (incf (clause-pvs clause)) 2)
		   (when (setq node2 (clause-consequent clause))
		     (push node2 new)))))
	(:FALSE (dolist (clause (tms-node-true-clauses node))
		  (when (= (incf (clause-pvs clause)) 2)
		    (when (setq node2 (clause-consequent clause))
		      (push node2 new)))))))))

(defun clause-consequent (clause)
  (dolist (term-pair (clause-literals clause))
    (when (eq (tms-node-label (car term-pair)) (cdr term-pair))
      (return (if (eq clause (tms-node-support (car term-pair)))
		  (car term-pair))))))

(defun clause-antecedents (clause &aux (antes nil)
				  (used? nil))
  (dolist (term-pair (clause-literals clause)
		     (if used? antes))
	  (cond ((eq (tms-node-support (car term-pair))
		     clause) (setq used? t))
		(t (push (car term-pair) antes)))))

(defun find-alternative-support (ltms nodes)
  (dolist (node nodes)
    (when (unknown-node? node)
      (check-clauses ltms (tms-node-true-clauses node))
      (check-clauses ltms (tms-node-false-clauses node))))
  (if (ltms-complete ltms) (lazy-pi nodes ltms)))

;;; Contradiction handling interface.
(defun check-for-contradictions (ltms &aux violated-clauses)
  (setq violated-clauses
	(delete-if-not #'violated-clause? (ltms-violated-clauses ltms)))
  (setf (ltms-violated-clauses ltms) nil)
  (if violated-clauses (contradiction-handler ltms violated-clauses)))

(defun contradiction-handler (ltms violated-clauses)
  (cond ((not (ltms-checking-contradictions ltms))
	 ;; Don't want to deal with it now, so save it for later.
	 (setf (ltms-pending-contradictions ltms)
	       (nconc violated-clauses (ltms-pending-contradictions ltms))))
	(t (dolist (handler (ltms-contradiction-handlers ltms))
	     (if (funcall handler violated-clauses ltms) (return T))))))

(defmacro without-contradiction-check (ltms &body body)
  (contradiction-check ltms nil body))

(defmacro with-contradiction-check (ltms &body body)
  (contradiction-check ltms t body))

(defun contradiction-check (ltms flag body)
  `(let* ((.ltms. ,ltms)
	  (.old-value. (ltms-checking-contradictions .ltms.)))
     (unwind-protect
	 (progn (setf (ltms-checking-contradictions .ltms.) ,flag)
		,@body)
       (setf (ltms-checking-contradictions .ltms.) .old-value.))))

(defmacro with-contradiction-handler (ltms handler &body body)
  `(let ((.ltms. ,ltms))
     (unwind-protect
	 (progn (push ,handler (ltms-contradiction-handlers .ltms.))
		,@ body)
       (pop (ltms-contradiction-handlers .ltms.)))))

(defmacro with-assumptions (assumption-values &body body)
  ;; Allows assumptions to be made safely, and retracted properly
  ;; even if non-local exits occur.
  `(unwind-protect (progn (dolist (av ,assumption-values)
			    (enable-assumption (car av) (cdr av)))
			 ,@ body)
     (dolist (av ,assumption-values) (retract-assumption (car av)))))

;;; Well-founded support inquiries.
(defun support-for-node (node &aux result support)
  (cond ((null (setq support (tms-node-support node))) nil)
	((eq support :ENABLED-ASSUMPTION) :ENABLED-ASSUMPTION)
	(t (dolist (pair support)
	     (unless (eq (car pair) node)
	       (push node result)))
	   (values result (clause-informant support)))))

(defun assumptions-of-node (node)
  (cond ((eq :ENABLED-ASSUMPTION (tms-node-support node)) (list node))
	((known-node? node) (assumptions-of-clause (tms-node-support node)))))

(defun assumptions-of-clause (in-clause &aux)
  (do ((clause-queue (list in-clause)
		     (nconc (cdr clause-queue) new-clauses))
       (mark (list nil))
       (node nil)
       (new-clauses nil nil)
       (assumptions nil))
      ((null clause-queue) assumptions)
    (dolist (term-pair (clause-literals (car clause-queue)))
      (setq node (car term-pair))
      (unless (eq (tms-node-mark node) mark)
	(unless (eq (tms-node-label node) (cdr term-pair))
	  (cond ((eq :ENABLED-ASSUMPTION (tms-node-support node))
		 (push node assumptions))
		((null (tms-node-support node)) (ltms-error "Node is unknown" node))
		(t (push (tms-node-support node) new-clauses))))
	(setf (tms-node-mark node) mark)))))

;;; Inference engine stub to allow the LTMS to be used stand alone.
(defun why-node (node)
  (cond ((unknown-node? node)
	 (format t "~%~A is unknown." (node-string node))
	 nil)
	((eq :ENABLED-ASSUMPTION (tms-node-support node))
	 (format t "~%~A is ~A <~A>"
		 (node-string node)
		 (tms-node-label node) (tms-node-support node))
	 nil)
	(t (format t "~%~A is ~A via ~A on"
		   (node-string node)
		   (tms-node-label node)
		   (clause-informant (tms-node-support node)))
	   (dolist (term-pair (clause-literals (tms-node-support node)))
	     (unless (equal (tms-node-label (car term-pair))
			    (cdr term-pair))
	       (format t "~%   ~A is ~A"
		       (node-string (car term-pair))
		       (tms-node-label (car term-pair)))))))
  node)

(defun why-nodes (ltms)
  (dolist (node (ltms-nodes ltms)) (why-node node)))

;;; This prints a well-founded explanation for a node.  
(defvar *line-count*)

(defun explain-node (node &aux *line-count*)
  (setq *line-count* 0)
  (dolist (node (ltms-nodes (tms-node-ltms node)))
    (setf (tms-node-mark node) nil))
  (explain-1 node))

(defun explain-1 (node &aux antecedents)
  (cond ((tms-node-mark node))
	((eq :ENABLED-ASSUMPTION (tms-node-support node))
	 (format T "~%~3D ~15<~:[(:NOT ~A)~;~A~]~>~15<()~>   Assumption"
		 (incf *line-count*) (true-node? node) (node-string node))
	 (setf (tms-node-mark node) *line-count*))
	(t (dolist (term-pair (clause-literals (tms-node-support node)))
	     (unless (equal (tms-node-label (car term-pair))
			    (cdr term-pair))
	       (push (explain-1 (car term-pair)) antecedents)))
	   (format T "~%~3D ~15<~:[(:NOT ~A)~;~A~]~> ~15<~A~>  "
		   (incf *line-count*) (true-node? node) (node-string node) antecedents)
	   (pretty-string-clause (tms-node-support node))
	   (setf (tms-node-mark node) *line-count*))))

(defun pretty-print-clauses (ltms)
  (dolist (l (ltms-clauses ltms))
    (format T "~% ")
    (pretty-string-clause l)))

(defun pretty-string-clause (clause)
  (format T "(:OR")
  (dolist (literal (clause-literals clause))
    (format T " ~:[(:NOT ~A~;~A~]"
	    (eq :TRUE (cdr literal)) (node-string (car literal))))
  (format T ")"))

;;;; More interrogatives

(defun show-node-consequences (node)
  (let ((conseqs (node-consequences node)))
    (cond (conseqs 
	   (format t "~% Consequences of ~A:" (signed-node-string node))
	   (dolist (conseq conseqs)
		   (format t "~%  ~A" (signed-node-string conseq))))
	  (t (format t "~% ~A has no consequences." (node-string node))))))

(defun signed-node-string (node)
  (if (true-node? node) (node-string node)
    (if (false-node? node) (format nil "Not[~A]" (node-string node))
      (format nil "Unknown[~A]" (node-string node)))))

(defun node-consequences (node &aux conseq conseqs)
  ;; Nodes known due to clauses which this fact
  ;; participates in.
  (dolist (cl (tms-node-true-clauses node))
	  (unless (eq cl (tms-node-support node))
		  ;; Not yourself, dummy
		  (setq conseq (clause-consequent cl))
		  (if conseq (push conseq conseqs))))
  (dolist (cl (tms-node-false-clauses node))
	  (unless (eq cl (tms-node-support node))
		  ;; Not yourself, dummy
		  (setq conseq (clause-consequent cl))
		  (if conseq (push conseq conseqs))))
  conseqs)

(defun listify-clause (cl)
  (cons ':OR
	(mapcar #'(lambda (pair)
		    (if (eq (cdr pair) ':FALSE)
			(list ':NOT (node-string (car pair)))
		      (node-string (car pair))))
		(clause-literals cl))))

(defun node-show-clauses (node)
  (format t "For ~A:" (node-string node))
  (dolist (cl (tms-node-true-clauses node))
	  (pprint (listify-clause cl)))
  (dolist (cl (tms-node-false-clauses node))
	  (pprint (listify-clause cl)))
  node)

;;;; Interactive wandering in the dependency network

(defun wander-dnet (node)
  (unless (known-node? node)
	  (format t "~% Sorry, ~A not believed." (view-node node))
	  (return-from wander-dnet node))
  (do ((stack nil)
       (current node)
       (mode ':ante)
       (options nil)
       (olen 0)
       (done? nil))
      (done? current)
      (cond ((eq mode ':ante)
	     (why-node current)
	     (setq options (if (typep (tms-node-support current) 'clause)
			       (clause-antecedents (tms-node-support current))
			     nil)))
	    (t ;; Looking at consequences
	     (show-node-consequences current)
	     (setq options (node-consequences current))))
      (setq olen (length options))
      (do ((good? nil)
	   (choice 0))
	  (good? (case good?
		       (Q (return-from wander-dnet current))
		       (c (setq mode ':conseq))
		       (a (setq mode ':ante))
		       (0 (if stack
			      (setq current (pop stack))
			    (return-from wander-dnet current)))
		       (t (push current stack)
			  (setq current (nth (1- good?) options)))))
	  (format t "~%>>>")
	  (setq choice (read))
	  (cond ((or (eq choice 'q)
		     (eq choice 'c)
		     (eq choice 'a)
		     (and (integerp choice)
			  (not (> choice olen))
			  (not (< choice 0))))
		 (setq good? choice))
		(t (format t "~% Must be q, a, c or an integer from 0 to ~D." olen))))))

(proclaim '(special *contra-assumptions*))

(defun ask-user-handler (contradictions ltms)
  (declare (ignore ltms))
  (dolist (contradiction contradictions)
    (if (violated-clause? contradiction)
	(handle-one-contradiction contradiction))))

(defun handle-one-contradiction (violated-clause)
  (let ((*contra-assumptions* (assumptions-of-clause violated-clause))
	(the-answer nil))
    (unless *contra-assumptions* (ltms-error "Global contradiction"
					     violated-clause))
    (format t "~%Contradiction found:")
    (print-contra-list *contra-assumptions*)
    (format t "~%Call (TMS-ANSWER <number>) to retract assumption.")
    (setq the-answer
	  (catch 'tms-contradiction-handler
	    (break "LTMS contradiction break")))
    (if (and (integerp the-answer)
	     (> the-answer 0)
	     (not (> the-answer (length *contra-assumptions*))))
	(retract-assumption (nth (1- the-answer)
				 *contra-assumptions*)))))

(defun print-contra-list (nodes)
  (do ((counter 1 (1+ counter))
       (nn nodes (cdr nn)))
      ((null nn))
    (format t "~%~A ~A" counter
	    (node-string (car nn)))))

(defun tms-answer (num)
  (if (integerp num)
      (if (> num 0)
	  (if (not (> num (length *contra-assumptions*)))
	      (throw 'tms-contradiction-handler num)
	      (format t "~%Ignoring answer, too big."))
	  (format t "~%Ignoring answer, too small"))
      (format t "~%Ignoring answer, must be an integer.")))

;;; Use this to get rid of all contradictions.
(defun avoid-all (contradictions ltms &aux culprits culprit sign)
  ltms
  (dolist (contradiction contradictions)
    (when (violated-clause? contradiction)
      (unless (setq culprits (assumptions-of-clause contradiction))
	(ltms-error "Total contradiction" contradiction))
      (setq culprit (car culprits)
	    sign (tms-node-label culprit))
      (retract-assumption culprit)
      (add-nogood culprit sign culprits)
      t)))

