;; -*- lisp -*- 

;;;; Variables and pattern matching -- LTRE
;;;; Version 8, 5/6/91

;;; Copyright 1986, 1989, 1990, 1991 Kenneth D. Forbus, Northwestern University,
;;; and Johan de Kleer, Xerox Corporation.  All Rights Reserved.

;;; Since the reference mechanism is independent of the belief
;;; mechanism, this code is essentially identical to previous versions.

(in-package 'user)

(defun variable? (x)
  (and (symbolp x)	;A symbol whose first character is "?"
       (char= #\? (elt (symbol-name x) 0))))

(defun quotize (pattern)
  (cond ((null pattern) nil)
	((variable? pattern) pattern)
	((not (listp pattern)) (list 'QUOTE pattern))
	((eq (car pattern) ':EVAL) (cadr pattern))
	(t `(cons ,(quotize (car pattern))
		  ,(quotize (cdr pattern))))))

(defun pattern-free-variables (pattern)
  (pattern-free-vars1 pattern nil))

(defun pattern-free-vars1 (pattern vars)
  (cond ((null pattern) vars)
	((variable? pattern)
	 (if (or (member pattern vars)
		 (member pattern *bound-vars*))
	     vars
	     (cons pattern vars)))
	((atom pattern) vars)
	(t (pattern-free-vars1
	     (cdr pattern)
	     (pattern-free-vars1 (car pattern) vars)))))

(defmacro rlet (var-specs &rest body)
  ;; Provides means for lisp code in body to
  ;; add information to the rule's environment.
  (let ((*bound-vars* (append (mapcar #'car var-specs) *bound-vars*)))
    `(let ,(mapcar #'(lambda (let-clause)
				(list (car let-clause)
				      (if (and (listp (cadr let-clause))
					       (eq (car (cadr let-clause))
						   ':EVAL))
					  (cadr (cadr let-clause))
					  (quotize (cadr let-clause)))))
			    var-specs)
       ,@ (fully-expand-body body))))

;;; Our old friend

(defun unify (a b) (unify1 a b nil))

(defun unify1 (a b bindings)
   (cond ((equal a b) bindings)
	 ((variable? a) (unify-variable a b bindings))
	 ((variable? b) (unify-variable b a bindings))
	 ((or (atom a) (atom b)) ':FAIL)
	 ((not
	    (eq ':FAIL (setq bindings
			    (unify1 (car a) (car b) bindings))))
	  (unify1 (cdr a) (cdr b) bindings))
	 (t ':FAIL)))

(defun unify-variable (var exp bindings &aux val)
  (setq val (cdr (assoc var bindings)))
  (cond (val (unify1 val exp bindings))
	((free-in? var exp bindings)
	 (cons (cons var exp) bindings))
	(t ':FAIL)))

(defun free-in? (var exp bindings)
  (cond ((null exp) t)
	((equal var exp) nil)
	((variable? exp)
	 (free-in? var (cdr (assoc exp bindings)) bindings))
	((atom exp) t)
	((free-in? var (car exp) bindings)
	 (free-in? var (cdr exp) bindings))))

;;;; Open-coding unification

;; Construct a defun specialized to match the given pattern.
;; That function will return NIL if no match,
;;   (values T <binding-spec> <node flag>) if match is successful.
;;  <node flag> is non-nil whenever the TMS node should be included in
;;  the rule entry for future testing to match belief conditions.
;;  Notice that arguments corresponding to :TEST and :VAR options must be
;;   passed in, too. 

(defun generate-match-procedure (pattern var test condition)
  (multiple-value-bind (tests binding-specs)
       ;;make special tests to check for this pattern
        (generate-match-body pattern (pattern-free-variables pattern) test)
    `(defun ,(generate-rule-procedure-name pattern) (P ,@ *bound-vars*)
       ;;first arg, P, is the pattern
       (if (and ,@ tests)
	   (values T (list ,@ (if var '(P)) ,@ (reverse binding-specs))
		   ,(unless (eq condition ':INTERN) t))))))

(defun generate-match-body (pattern vars extra-test
				    &aux structure-tests var-alist
				    equal-tests binding-specs)
  (dolist (test (generate-unify-tests pattern vars nil 'P)
		(values (append structure-tests equal-tests
				(if extra-test
				    (list (sublis var-alist
						  extra-test))))
			binding-specs))
    (cond ((variable? (car test))
	   ;test looks like (?x (nth p) (nth p) ...)
	   (setq equal-tests
		 (append (generate-pairwise-tests (cdr test))
			 equal-tests))
   ;; due to "push", last represents the first instance of var
	   (if extra-test 
	       (push (cons (car test) (car (last test))) var-alist))
	   (push (car (last test)) binding-specs))
	  (t (push test structure-tests)))))

(defun generate-pairwise-tests (tests)
  (cond ((or (null tests) (null (cdr tests))) nil)
	(t (cons (list 'EQUAL (car tests) (cadr tests))
		 (generate-pairwise-tests (cdr tests))))))

;;; Generate a list of explicit tests for matching 
;;; the given pattern. Assumes that the pattern
;;;    to be tested will be in variable "P".
;;; Tests are returned in backward order.
;;; (generate-unify-tests '(foo ?x) nil nil 'P)
;;;     returns:    '((NULL (CDR (CDR P)))
;;;               (EQUAL ?X (CAR (CDR P)))
;;;                   (CONSP (CDR P))
;;;                   (EQUAL (QUOTE FOO) (CAR P))
;;;                   (CONSP P))
;;;
(defun generate-unify-tests (pattern vars tests path)
  (cond ((null pattern)
	 	;this is the end
	 (cons `(null ,path) tests))
	((member pattern vars)	                  
         ;; must see if the pattern has been bound elsewhere,
	 ;; and if it has, test to see if the element here is
         ;; consistent with that earlier binding.
	 (let ((previous (assoc pattern tests)))
	   (cond (previous ;add this position to test it
		  (push path (cdr previous))
		  tests)
		 (t (cons (list pattern path) tests)))))
	;; if variable, it must be bound so test
	;; against the current value.
	((variable? pattern) (cons `(equal ,pattern ,path)
				   tests))
	;; if not a list, then see if equal
	((numberp pattern)
	 (cons `(and (numberp ,path) (= ,pattern ,path))
	       tests))
	((atom pattern) (cons `(equal ',pattern ,path) tests))
	;; recurse on a list
	(t (generate-unify-tests (cdr pattern) vars
		 (generate-unify-tests (car pattern) vars
				       ;avoid lisp errors
				       (cons `(consp ,path)
					     tests)
				       	    ;extend the path
				       (list 'car path))
		 ;extend path in other direction
		 (list 'cdr path)))))		



