/*******************************************************************************
+
+  LEDA 3.5
+
+  ray.h
+
+  This file is part of the LEDA research version (LEDA-R) that can be 
+  used free of charge in academic research and teaching. Any commercial
+  use of this software requires a license which is distributed by the
+  LEDA Software GmbH, Postfach 151101, 66041 Saarbruecken, FRG
+  (fax +49 681 31104).
+
+  Copyright (c) 1991-1997  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 66123 Saarbruecken, Germany     
+  All rights reserved.
+ 
*******************************************************************************/
#ifndef LEDA_RAY_H
#define LEDA_RAY_H

#if !defined(LEDA_ROOT_INCL_ID)
#define LEDA_ROOT_INCL_ID 350116
#include <LEDA/REDEFINE_NAMES.h>
#endif


#include <LEDA/point.h>
#include <LEDA/segment.h>

//------------------------------------------------------------------------------
// straight rays
//------------------------------------------------------------------------------


class ray_rep : public handle_rep {

friend class ray;
friend class line;
friend class circle;

  segment  seg; 

public:
   
  ray_rep() {}
  ray_rep(const segment& s) : seg(s) {}

 ~ray_rep() {}

friend inline int cmp_slopes(const ray&, const ray&);
friend inline int orientation(const ray&, const point&);

};
   
/*{\Manpage {ray} {} {Straight Rays}}*/

class ray   : public handle_base 
{

friend class line;
friend class circle;

/*{\Mdefinition
An instance $r$ of the data type $ray$ is a directed straight ray
in the two-dimensional plane. The angle between a right oriented horizontal
ray and $r$ is called the direction of $r$.}*/

ray_rep* ptr() const { return (ray_rep*)PTR; }

public:

/*{\Mcreation r }*/

 ray(const point& p, const point& q);
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to the 
ray starting at point $p$ and passing through point $q$. }*/


 ray(const segment& s);
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to 
$ray(s.source(),s.target())$. }*/

 ray(const point& p, const vector& v);
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to
$ray(p,p+v)$.}*/

 ray(const point& p, double alpha);
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to the 
ray starting at point $p$ with direction $alpha$.}*/

 ray();
/*{\Mcreate 
introduces a variable |\Mvar| of type |\Mname|. |\Mvar| is initialized to the 
ray starting at the origin with direction 0.}*/


 ray(const ray& r) : handle_base(r) {};
 ray& operator=(const ray& r) { handle_base::operator=(r); return *this; }
~ray() {}



/*{\Moperations 2 4.5 }*/


point  source() const   { return ptr()->seg.source(); }
/*{\Mop     returns the source of |\Mvar|.}*/

point  point1() const   { return ptr()->seg.source(); }
/*{\Mop     returns the source of |\Mvar|.}*/

point  point2() const   { return ptr()->seg.target(); }
/*{\Mop     returns a point on |\Mvar| different from |\Mvar.source()|.}*/

double direction() const     { return ptr()->seg.angle(); }
/*{\Mop     returns the direction of |\Mvar|.}*/

double angle(const ray& s) const { return ptr()->seg.angle(s.ptr()->seg); }
/*{\Mop     returns the angle between |\Mvar| and $g$, i.e., 
	    $g$.direction() $-$ |\Mvar|.direction().}*/

bool is_vertical() const    { return ptr()->seg.is_vertical();  }
/*{\Mop     returns true iff |\Mvar| is vertical.}*/

bool is_horizontal() const  { return ptr()->seg.is_horizontal();}
/*{\Mop     returns true iff |\Mvar| is horizontal.}*/


double slope() const     { return ptr()->seg.slope();     }
/*{\Mop     returns the slope of |\Mvar|.\\
	    \precond  |\Mvar|  is not vertical.}*/

segment seg()  const     { return ptr()->seg; }


bool intersection(const ray& s, point& inter) const;
/*{\Mopl    if $r$ and $s$ intersect in a single point this point 
            is assigned to $inter$ and the result is |true|, otherwise 
            the result is |false|.}*/

bool intersection(const segment& s, point& inter) const;
/*{\Mopl    if $r$ and $s$ intersect in a single point this point 
            is assigned to $inter$ and the result is |true|, otherwise 
            the result is |false|.}*/

ray translate_by_angle(double a, double d) const 
{ return ptr()->seg.translate_by_angle(a,d); }
/*{\Mopl     returns |\Mvar| translated in direction $a$ by distance $d$.}*/

ray translate(double dx, double dy) const 
{ return ptr()->seg.translate(dx,dy); }
/*{\Mopl     returns |\Mvar| translated by vector $(dx,dy)$.}*/

ray translate(const vector& v)  const 
{ return ptr()->seg.translate(v); }
/*{\Mopl     returns |\Mvar| translated by vector $v$\\
	    \precond $v$.dim() = 2.}*/ 

ray  operator+(const vector& v) const { return translate(v); }
/*{\Mbinop   returns |\Mvar| translated by vector $v$.}*/

ray  operator-(const vector& v) const { return translate(-v); }
/*{\Mbinop   returns |\Mvar| translated by vector $-v$.}*/


ray  rotate(const point& q, double a) const
{ return ptr()->seg.rotate(q,a); }
/*{\Mopl    returns |\Mvar| rotated about point $q$ by angle $a$.}*/

ray  rotate(double a) const  
{ return rotate(point(0,0),a);}
/*{\Mop     returns $r$.rotate($point(0,0),\ a$). }*/

ray  rotate90(const point& q) const
{ return ptr()->seg.rotate90(q); }
/*{\Mopl    returns |\Mvar| rotated about $q$ by angle of $90$ degrees.}*/


ray  reflect(const point& p, const point& q) const
{ return ptr()->seg.reflect(p,q); }
/*{\Mop     returns |\Mvar| reflected  across the straight line passing
            through $p$ and $q$.}*/

ray  reflect(const point& p) const
{ return ptr()->seg.reflect(p); }
/*{\Mop     returns |\Mvar| reflected  across point $p$.}*/


bool contains(const point&) const;
/*{\Mopl    decides whether |\Mvar| contains $p$. }*/

bool contains(const segment&) const;
/*{\Mopl    decides whether |\Mvar| contains $s$. }*/


bool operator==(const ray& s) const;
bool operator!=(const ray& s) const { return !operator==(s); }



/*{\Mtext
{\bf Non-Member Functions}
\smallskip
}*/


friend int orientation(const ray& r, const point& p);
/*{\Mfunc      computes orientation($a$, $b$, $p$), where $a \not= b$
and $a$ and $b$ appear in this order on ray $r$. }*/

friend int cmp_slopes(const ray& r1, const ray& r2);
/*{\Mfunc      returns compare(slope($r_1$), slope($r_2$)).}*/


friend ostream& operator<<(ostream& out, const ray& r);
friend istream& operator>>(istream& in, ray& r);  

};

inline  int orientation(const ray& r, const point& p)
{ return orientation(r.ptr()->seg,p); }

inline  int cmp_slopes(const ray& r1, const ray& r2)
{ return cmp_slopes(r1.ptr()->seg,r2.ptr()->seg); }

inline bool parallel(const ray& r1, const ray& r2)
{ return cmp_slopes(r1,r2) == 0; }



inline char* leda_tname(const ray*) { return "ray"; }

#if LEDA_ROOT_INCL_ID == 350116
#undef LEDA_ROOT_INCL_ID
#include <LEDA/UNDEFINE_NAMES.h>
#endif

#endif
