
(add-to-list 'load-path "~/.emacs.d/el-get/el-get")

(unless (require 'el-get nil 'noerror)
  (with-current-buffer
      (url-retrieve-synchronously
       "https://raw.github.com/dimitri/el-get/master/el-get-install.el")
    (let (el-get-master-branch)
      (goto-char (point-max))
      (eval-print-last-sexp))))

;; (el-get 'sync)

(when (version< emacs-version "24.3")
  (el-get 'sync '(cl-lib))
  (add-to-list 'load-path "~/.emacs.d/el-get/cl-lib"))

(setq ome-packages nil)

(defun ome-load (file &optional header-or-tag)
  "Load configuration from other ome-*.org files.
If the optional argument is the id of a subtree then only
configuration from within that subtree will be loaded.  If it is
not an id then it will be interpreted as a tag, and only subtrees
marked with the given tag will be loaded.

For example, to load all of ome-lisp.org simply
add (ome-load \"lisp\") to your configuration.

To load only the 'window-system' config from
ome-misc-recommended.org add
 (ome-load \"misc-recommended\" \"window-system\")
to your configuration."
  (let ((file (expand-file-name (if (string-match ".+\.org" file)
                                    file
                                  (format "%s.org" file))
                                ome-dir)))
    ;; ensure el-get-sources is empty before loading "ome-.+\.org" files
    (setq el-get-sources nil)
    (org-babel-load-file
     (if header-or-tag
         (let* ((base (file-name-nondirectory file))
                (dir  (file-name-directory file))
                (partial-file (expand-file-name
                               (concat "." (file-name-sans-extension base)
                                       ".part." header-or-tag ".org")
                               dir)))
           (unless (file-exists-p partial-file)
             (with-temp-file partial-file
               (insert
                (with-temp-buffer
                  (insert-file-contents file)
                  (save-excursion
                    (condition-case nil ;; collect as a header
                        (progn
                          (org-link-search (concat"#"header-or-tag))
                          (org-narrow-to-subtree)
                          (buffer-string))
                      (error ;; collect all entries with as tags
                       (let (body)
                         (org-map-entries
                          (lambda ()
                            (save-restriction
                              (org-narrow-to-subtree)
                              (setq body (concat body "\n" (buffer-string)))))
                          header-or-tag)
                         body))))))))
           partial-file)
       file))
    (el-get 'sync (mapcar 'el-get-source-name el-get-sources))
    (mapcar (lambda (el-get-package)
              (add-to-list 'ome-packages
                           (el-get-source-name el-get-package)))
            el-get-sources)))

(defun ome-try-get-package-website (package)
  "el-get's package recipe has multiple type, some contains
a :website, some contains just a :url, while some github package
just contains a :pkgname. This function try to get a proper
website link for an el-get package."
  (let ((package-def (el-get-package-def package)))
    (or (plist-get package-def :website)
        (and (eq (plist-get package-def :type) 'github)
             (concat "https://github.com/"
                     (plist-get package-def :pkgname)))
        (plist-get package-def :url))))

(defun ome-package-list (ome-packages)
  "Get a whole list of el-get packages installed by
oh-my-emacs. Returns a list of (package-name, package-website,
package-description)."
  (mapcar
   (lambda (package)
     (let ((package-def (el-get-package-def package)))
       (list (ome-try-get-package-website package)
             (el-get-package-name package)
             ;; some package's description has multiple lines, so we need to
             ;; join them together for better auto-fill.
             (replace-regexp-in-string "\\(\n\\|\\ \\)+" " "
                                       (plist-get package-def :description)))))
   (sort ome-packages #'string<)))

(defun ome-package-list-in-org-format (ome-packages)
  "A convenient function for constructing oh-my-emacs's
README.org file."
  (mapcar
   (lambda (ome-package)
     (format "[[%s][%s]]: %s"
             (nth 0 ome-package)
             (nth 1 ome-package)
             (nth 2 ome-package)))
   (ome-package-list ome-packages)))

(defun ome-insert-package-list-in-org-mode ()
  "Insert all packages info to current `org-mode' files.

This still has some problems with `org-mode' and `auto-fill-mode'."
  (interactive)
  (insert
   (with-temp-buffer
     (loop for package
           in (ome-package-list-in-org-format ome-packages)
           do (insert "- " package "\n\n"))
     (fill-region (point-min) (point-max))
     (replace-string "-\n " "-" nil (point-min) (point-max))
     (flush-lines "^$" (point-min) (point-max))
     (buffer-string))))

(defun ome-install (el-get-package)
  "Add EL-GET-PACKAGE to `el-get-sources'.

This package will be installed when `ome-load'. Users can make
his own customization by providing a \"ome-package-name-setup\"
function."
  (let ((ome-package-setup-func
         (intern
          (concat "ome-"
                  (el-get-as-string el-get-package)
                  "-setup"))))
    (if (fboundp ome-package-setup-func)
        (add-to-list 'el-get-sources
                     `(:name ,el-get-package
                             :after (progn
                                      (,ome-package-setup-func))))
      (add-to-list 'el-get-sources
                   `(:name ,el-get-package)))))

(add-to-list 'el-get-sources
             '(:name cl-lib))

(defun ome-start-or-switch-to (function buffer-name)
  "Invoke FUNCTION if there is no buffer with BUFFER-NAME.
  Otherwise switch to the buffer named BUFFER-NAME.  Don't clobber
  the current buffer."
  (if (not (get-buffer buffer-name))
      (progn
        (split-window-sensibly (selected-window))
        (other-window 1)
        (funcall function))
    (switch-to-buffer-other-window buffer-name)))

(ome-load "mu.org")
  ;(ome-load "mu-basic.org")
  ;(ome-load "mu-gui.org")
  ;(ome-load "mu-matlab.org")
  ;(ome-load "mu-shell.org")
  ;(ome-load "mu-edit.org")
  ;(ome-load "mu-keybindings.org")
  ;  (ome-load "ome-gui.org")
  ;  (ome-load "ome-basic.org")
  ;  (ome-load "ome-miscs.org")
  ;  (ome-load "ome-completion.org")
  ;  (ome-load "ome-keybindings.org")
    (ome-load "ome-org.org")
  ;  (ome-load "ome-advanced.org")
  ;  (ome-load "ome-writing.org")
  ;  (ome-load "ome-lisp.org")
  ;  (ome-load "ome-python.org")
  ;  (ome-load "ome-auto-mode.org")
  ;  (ome-load "ome-tex.org")

  (add-to-list 'load-path (expand-file-name "~/.emacs.d/"))
;  (load "mu-latex")
;  (ome-load "mu-cc.org")

(flet ((sk-load (base)
         (let* ((path          (expand-file-name base ome-dir))
                (literate      (concat path ".org"))
                (encrypted-org (concat path ".org.gpg"))
                (plain         (concat path ".el"))
                (encrypted-el  (concat path ".el.gpg")))
           (cond
            ((file-exists-p encrypted-org) (org-babel-load-file encrypted-org))
            ((file-exists-p encrypted-el)  (load encrypted-el))
            ((file-exists-p literate)      (org-babel-load-file literate))
            ((file-exists-p plain)         (load plain)))))
       (remove-extension (name)
         (string-match "\\(.*?\\)\.\\(org\\(\\.el\\)?\\|el\\)\\(\\.gpg\\)?$" name)
         (match-string 1 name)))
  (let ((elisp-dir (expand-file-name "src" ome-dir))
        (user-dir (expand-file-name user-login-name ome-dir)))
    ;; add the src directory to the load path
    (add-to-list 'load-path elisp-dir)
    ;; load specific files
    (when (file-exists-p elisp-dir)
      (let ((default-directory elisp-dir))
        (normal-top-level-add-subdirs-to-load-path)))
    ;; load system-specific config
    (sk-load system-name)
    ;; load user-specific config
    (sk-load user-login-name)
    ;; load any files in the user's directory
    (when (file-exists-p user-dir)
      (add-to-list 'load-path user-dir)
      (mapc #'sk-load
            (remove-duplicates
             (mapcar #'remove-extension
                     (directory-files user-dir t ".*\.\\(org\\|el\\)\\(\\.gpg\\)?$"))
             :test #'string=)))))

(setq custom-file (concat ome-dir "custom.el"))
(load custom-file 'noerror)
