/* audio.js -- web audio support functions for GUI side
 *
 * Roger B. Dannenberg
 * May 2023
 *
 * Simple use is:
 *     audio_need_input(true);  // or false -- do you need input?
 *     add_worklet_module(filename, initfn);  // call for each wasm module
 *          // you need. filename is the path to the emscripten-generated
 *          // *.js file. initfn is an exported function with signature
 *          // initfn(int i). It should create the audio worklet using
 *          // window.emgl_audio_context and it should immediately copy
 *          // window.emgl_audio_worklet_callback in order to call it later,
 *          // passing in the value of i, which is an index used to keep
 *          // track of which audio_worklets are initialized.
 *     await audio_initialized();  // wait until everything is running
 *     audio_node_create(name, ins, outs, options);  // returns a node
 *          // the name is the filename passed to add_worklet_module()
 *     audio_input can be read after audio_initialized() returns
 * Import with:
 *
import {audio_need_input, audio_worklet_module, audio_initialized,
        audio_node_create, audioContext, audio_input} from "./audio.js";
 *
 */

export let audioContext = null;
export let audio_input = null;

let audio_ready = null;  // a promise of initialization
let audio_initialization_started = false;  // started yet?
let audio_open_input_flag = false;  // do we need to open input?
const worklet_modules = [];  // names of worklet files to load


// returns a promise that is fulfilled when audio is initialized.
// this will start audio initialization if not yet started, and
// always returns a promise.
export function audio_initialized() {
    if (!audio_initialization_started) {
        audio_ready = audio_initialization();
        audio_initialization_started = true;
    }
    return audio_ready;
}    


export function audio_need_input(flag) {
    audio_open_input_flag = flag;
}


export function add_worklet_module(jsfilename, init_function) {
    worklet_modules.push([jsfilename, init_function, false]);
}


async function wait_for_all_worklets() {
    // note: this uses polling where await might be cleaner, but I do not
    // know that promises and resolve and other mechanisms will work across
    // threads. JavaScript seems to be based on a single-thread model, and
    // Worklets seems to be a hack to break out of the single-thread
    // restriction, so is it safe to mix-and-match features of both?
    let done = false;
    while (!done) {
        done = true;  // assume done unless we find a worklet not done
        for (let i = 0; i < worklet_modules.length; i++) {
            if (!worklet_modules[i][2]) {
                done = false;
            }
        }
        await new Promise(r => setTimeout(r, 50));  // sleep 50ms
    }
    return;
}


// This function is passed into emscripten code via
// window.emgl_audio_worklet_callback and is called with index i and the
// created worklet when the worklet is created.
function worklet_callback(i, worklet)
{
    worklet_modules[i][2] = worklet;
}


async function audio_initialization(need_input) {
    // const audio = new AudioContext();
    const AudioContextConstructor =
            window.AudioContext || window.webkitAudioContext; 
    audioContext = new AudioContextConstructor();
    console.log("audio_initialization: audioContext", audioContext);
    // put audioContext where emscripten EM_JS code can find it:
    window.emgl_audio_context = audioContext;
    window.emgl_worklet_modules = worklet_modules;
    // I think this loads sequentially, but probably we are not loading much
    for (let i = 0; i < worklet_modules.length; i++) {
        // pass in a function to be called when the worklet is created and
        // initialized
        window.emgl_audio_worklet_callback = worklet_callback
        let [jsfilename, worklet_init_function, done] = worklet_modules[i]
        worklet_init_function(i)
    }
    wait_for_all_worklets()
    if (audio_open_input_flag) {
        const constraints = { audio:true };
        const input_stream =
                await navigator.mediaDevices.getUserMedia(constraints);
        audio_input = audioContext.createMediaStreamSource(input_stream);
    }
    return true;
}


export function audio_node_create(name, ins, outs, options = []) {
    const opts = {"numberOfInputs": ins == 0 ? 0 : 1,
                  "numberOfOutputs": outs == 0 ? 0 : 1,
                  "outputChannelCount": outs == 0 ? [] : [outs],
                  "processorOptions": options};
    const node = new AudioWorkletNode(audioContext,  name, opts);
    return node;
}
