#include <emscripten/webaudio.h>
#include <emscripten/bind.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <assert.h>

/* Steps to use Wasm-based AudioWorklets:
  1. Create a Web Audio AudioContext either via manual JS code and calling emscriptenRegisterAudioObject() from JS, or by calling emscripten_create_audio_context() (shown in this sample)
  2. Initialize a Wasm AudioWorklet scope on the audio context by calling emscripten_start_wasm_audio_worklet_thread_async(). This shares the Wasm Module, Memory, etc. to the AudioWorklet scope,
     and establishes the stack space for the Audio Worklet.
     This needs to be called exactly once during page's lifetime. There is no mechanism in Web Audio to shut down/uninitialize the scope.
  3. Create one or more of Audio Worklet Processors with the desired name and AudioParam configuration.
  4. Instantiate Web Audio audio graph nodes from the above created worklet processors, specifying the desired input-output configurations and Wasm-side function callbacks to call for each node.
  5. Add the graph nodes to the Web Audio graph, and the audio callbacks should begin to fire.
*/

// a global where we save the context for use when we create an audio node:
EMSCRIPTEN_WEBAUDIO_T gl_audio_context = NULL;

#define NAME "tone"

#ifdef REPORT_RESULT // This is defined when running in Emscripten test harness. You can strip these out in your own project.
_Thread_local int testTlsVariable = 1;
int lastTlsVariableValueInAudioThread = 1;
#endif

typedef struct {
    bool stop;
    bool stopped;
    float gain;
    float phase;
    float phase_inc;
} Node_state;

// This function will be called for every fixed 128 samples of audio to be
// processed.
EM_BOOL ProcessAudio(int numInputs, const AudioSampleFrame *inputs,
                     int numOutputs, AudioSampleFrame *outputs, int numParams,
                     const AudioParamFrame *params, void *userData)
{
#ifdef REPORT_RESULT
    assert(testTlsVariable == lastTlsVariableValueInAudioThread);
    ++testTlsVariable;
    lastTlsVariableValueInAudioThread = testTlsVariable;
    assert(emscripten_current_thread_is_audio_worklet());
#endif
    Node_state *ns = (Node_state *) userData;

    // Produce tone in output channel 0.
    for (int j = 0; j < 128; ++j) {
        outputs[0].data[j] = sin(ns->phase) * ns->gain;
        ns->phase = fmod(ns->phase + ns->phase_inc, M_PI * 2);
    }
    // copy all k samples from output 0 channel 0 to all outputs i channels j
    for (int i = 0; i < numOutputs; i++) {
        for (int j = 1; j < outputs[i].numberOfChannels; j++) {
            for (int k = 0; k < 128; k++) {
                outputs[i].data[j * 128 + k] = outputs[0].data[k];
            }
        }

    }
    // We generated audio and want to keep this processor going.
    // Return EM_FALSE here to shut down.
    ns->stopped = ns->stop;  // let world know this state and node is free now
    return !ns->stop;
}


EM_JS(void, call_worklet_callback, (EMSCRIPTEN_WEBAUDIO_T audioContext,
                                    EM_BOOL success), {
    audioContext = emscriptenGetAudioObject(audioContext);
    window.emgl_audio_context = audioContext;
    window.emgl_audio_worklet_callback(success ? 0 : 1);
});


EM_JS(void, return_audio_node, (EMSCRIPTEN_AUDIO_WORKLET_NODE_T node), {
    node = emscriptenGetAudioObject(node);
    console.log("return_audio_node", node);
    window.emgl_audio_node = node;
});
      

#ifdef REPORT_RESULT
EM_BOOL main_thread_tls_access(double time, void *userData)
{
  // Try to mess the TLS variable on the main thread, with the expectation that it should not change
  // the TLS value on the AudioWorklet thread.
  testTlsVariable = (int)time;
  if (lastTlsVariableValueInAudioThread >= 100)
  {
    REPORT_RESULT(0);
    return EM_FALSE;
  }
  return EM_TRUE;
}
#endif

// This callback will fire after the Audio Worklet Processor has finished being
// added to the Worklet global scope.
void AudioWorkletProcessorCreated(EMSCRIPTEN_WEBAUDIO_T audioContext,
                                  EM_BOOL success, void *userData)
{
    call_worklet_callback(audioContext, success);
}


// This callback will fire when the Wasm Module has been shared to the
// AudioWorklet global scope, and is now ready to begin adding Audio
// Worklet Processors.
void WebAudioWorkletThreadInitialized(EMSCRIPTEN_WEBAUDIO_T audioContext,
                                      EM_BOOL success, void *userData)
{
    if (!success) return;

    WebAudioWorkletProcessorCreateOptions opts = { .name = NAME, };
    emscripten_create_wasm_audio_worklet_processor_async(audioContext, &opts,
                                             AudioWorkletProcessorCreated, 0);
}


// Define a global stack space for the AudioWorkletGlobalScope. Note that all
// AudioWorkletProcessors and/or AudioWorkletNodes on the given Audio Context
// all share the same AudioWorkerGlobalScope, i.e. they all run on the same
// one audio thread (multiple nodes/processors do not each get their own
// thread). Hence one stack is enough.
uint8_t wasmAudioWorkletStack[4096];

bool main_has_run = false;

int main()
{
    srand(time(NULL));

    assert(!emscripten_current_thread_is_audio_worklet());

    main_has_run = true;
}


// create the audio worklet thread
int worklet_create() {

    if (!main_has_run) {
        return 1;
    }

    EMSCRIPTEN_WEBAUDIO_T context = emscripten_create_audio_context(
                                     0 /* use default constructor options */);

    gl_audio_context = context;  // save it for use by audio_node_create

    emscripten_start_wasm_audio_worklet_thread_async(
            context, wasmAudioWorkletStack, sizeof(wasmAudioWorkletStack),
            WebAudioWorkletThreadInitialized, 0);

    return 0;
}


#define POLYPHONY 20
Node_state node_states[POLYPHONY];
int node_index = 0;

// Create an audio node instance.
// returns < 0 on error; otherwise returns a node instance id >= 0.
// The id could be used to find the node's state structure.
int audio_node_create() {
    // Specify the input and output node configurations for the Wasm Audio
    // Worklet. A simple setup with single mono output channel here,
    // and no inputs.
    int outputChannelCounts[1] = { 1 };

    EmscriptenAudioWorkletNodeCreateOptions options = {
            .numberOfInputs = 0,
            .numberOfOutputs = 1,
            .outputChannelCounts = outputChannelCounts
    };

    if (node_index >= POLYPHONY) return -2;  // too many nodes

    Node_state *node_state = &node_states[node_index];
    node_state->stop = false;
    node_state->stopped = false;
    node_state->gain = 0.1;
    node_state->phase = 0.0;
    // create some different frequencies depending on node_index:
    node_state->phase_inc = 220 * (node_index + 3) * M_PI * 2 / 48000;

    // Instantiate the noise-generator Audio Worklet Processor.
    EMSCRIPTEN_AUDIO_WORKLET_NODE_T wasmAudioWorklet =
            emscripten_create_wasm_audio_worklet_node(gl_audio_context,
                             NAME, &options, &ProcessAudio, node_state);

    // round-about way to return a value -- if we return it directly, an
    // integer is returned through the EMSCRIPTEN_BINDINGS method even if
    // we declare this function to be of type EMSCRIPTEN_AUDIO_WORKLET_NODE_T
    return_audio_node(wasmAudioWorklet);

    if (wasmAudioWorklet == NULL) {
        return -1;  // returns -1 on error
    } else {
        return node_index++;  // return this node's id
    }
}


void set_gain(float g) {
    // sine tone is already pretty piercing and continous too, so scale g by 0.1:
    node_states[0].gain = g * 0.1;
}


EMSCRIPTEN_BINDINGS(my_module) {
    emscripten::function("set_gain", &set_gain);
    emscripten::function("worklet_create", &worklet_create);
    emscripten::function("audio_node_create", &audio_node_create);
}
