/* audioutil.js -- web audio support functions for GUI side
 *
 * Roger B. Dannenberg
 * October 2023
 *
 * This is a modified incompatible version of audio.js, simplified to adapt
 * to my inability to pass an audio context into WASM code or to create an
 * audio worklet using WASM from JS.
 *
 * Simple use is:
 *     audio_need_input(true);  // or false -- do you need input?
 *     await audio_initialize(filename, initfn);  // load WASM module.
 *          // filename is the path to the emscripten-generated *.js file.
 *          // initfn is an exported function with one int parameter, 
 *          // which is 1 if audio input is needed, otherwise 0.
 *          // It should create the audio worklet and it should set
 *          // window.emgl_audio_context to the audio context and call
 *          // window.emgl_audio_worklet_callback with one int parameter
 *          // when the audio_worklet is initialized. The parameter is 0
 *          // on success and an error code to log otherwise. This code
 *          // is returned from audio_initialize, so if non-zero, assume
 *          // the audio_context is unavailable and audio_node_create will
 *          // not work.
 *     audio_node_create(name, ins, outs, options);  // returns a node and
 *          // its id in the form { audio_node, node_id }.
 *          // the name is the filename passed to audio_initialize(),
 *          // although at this point we only support one module.  Returns
 *          // null if there is an error, e.g. audio_initialize failed.
 *     audio_input can be read after audio_initialize() returns
 * Import with:
 *
import {audio_need_input, audio_initialize, audio_node_create, audio_input,
        audio_context} from "./audioutil.js";
 *
 */

export let audio_input = null;
export let audio_context = null;

let audio_ready = null;  // a promise of initialization
let audio_initialization_started = false;   // started yet?
let audio_worklet = null;  // the audio worklet created by WASM module
let audio_initialization_finished = false;  // finished yet?
let audio_open_input_flag = 0;  // do we need to open input? 0 or 1


// returns a promise that is fulfilled when audio is initialized.
// this will start audio initialization if not yet started, and
// always returns a promise.
export function audio_initialize(filename, initfn) {
    if (!audio_initialization_started) {
        audio_ready = audio_initialization(filename, initfn);
        audio_initialization_started = true;
    }
    return audio_ready;
}    


export function audio_need_input(flag) {
    audio_open_input_flag = flag ? 1 : 0;
}


async function wait_for_worklet() {
    // note: this uses polling where await might be cleaner, but I do not
    // know that promises and resolve and other mechanisms will work across
    // threads. JavaScript seems to be based on a single-thread model, and
    // Worklets seems to be a hack to break out of the single-thread
    // restriction, so is it safe to mix-and-match features of both?
    while (!audio_initialization_finished) {
        await new Promise(r => setTimeout(r, 50));  // sleep 50ms
    }
}


let audio_error_code = 0;

// This function is passed into emscripten code via
// window.emgl_audio_worklet_callback and is called
// when the worklet is created.
function worklet_callback(error_code)
{
    if (error_code != 0) {
        audio_error_code = error_code;
        console.warn("Something went wrong creating the audio worklet. " +
                     "Error code " + error_code);
    }
    console.log("WASM worklet return code " + error_code);
    audio_initialization_finished = true;
}


async function audio_initialization(filename, initfn) {
    window.emgl_audio_worklet_callback = worklet_callback
    console.log(Module, "initfn", initfn);
    let err = Module[initfn]()
    console.log("audio_initialization,", initfn, "returned", err);
    await wait_for_worklet()
    audio_context = window.emgl_audio_context
    if (audio_open_input_flag != 0) {
        const constraints = { audio:true };
        const input_stream =
                await navigator.mediaDevices.getUserMedia(constraints);
        audio_input = audio_context.createMediaStreamSource(input_stream);
    }
    return audio_error_code;
}


export function audio_node_create(name, ins, outs, options = []) {
    if (!audio_initialization_finished) {
        console.error("audio_node_create called before audio is initialized.");
        return null;
    } else if (audio_error_code) {
        console.error("audio initialization failed, so audio_node_create. " +
                      "cannot be called. Error code " + audio_error_code);
        return null;
    }

    let node_id = Module.audio_node_create();
    if (node_id < 0) {
        console.error("Module.audio_node_create returned", node_id);
    }
    // emscripten code places the created node here
    let audio_node = window.emgl_audio_node;
    window.emgl_audio_node = null;  // seems best not to leave it in a global
    // console.log("audio_node_create", {audio_node, node_id});
    return {audio_node, node_id};  // using object literal syntax extension
}
