/* awutil.cpp -- utility (helper) functions for audio worklets
 *
 * Roger B. Dannenberg
 * Oct 2023
 *
 * based on emscripten test code
 */

/* Functions to call:

From the WASM module main() function, call:
    register_audio_worklet_class(name, constructor)
with the name and a constructor (function pointer) that makes a new instance
of an Awnode subclass corresponding to name.

    main_has_run (boolean) must be set to true before returning from
main. This is an internal consistency check.

From JavaScript, call:

    worklet_create(). This returns 0 if the thread creation has
started, -1 on failure because the module's main_had_run global
variable is false (either you did not set it after running main and
registering worklet classes, or something else went wrong).  As a
side-effect, after the thread is created (and after this function
returns), window.emgl_audio_context is set to the audio context,
and then window.emgl_audio_worklet_callback() is called with 0 when
the worklet is created and the context is valid, or 1 to report
failure.

and call
    audio_node_create(name) after you get the
window.emgl_audio_worklet_callback() from worklet_create(). On
success, this returns the integer node id after setting
window.emgl_audio_node to the actual audio node. name is the string
name of the subclass of Awnode, matching a subclass that was
registered with register_audio_worklet_class().

From a node audio processing loop (the process() method), (optionally) call
    mono_to_all(numOutputs, outputs) to copy a single channel output
from output 0, channel 0, to all other outputs. In general, you would
compute each output channel individually rather than calling this,
which makes all outputs duplicates. (This is a method defined in awnode.h)

*/

#include "awutil.h"

#define MAX_NODES 128  // how many audio_node instances can be created
Awnode *audio_nodes[MAX_NODES];
int node_index = 0;

#define MAX_THREADS 8  // how many types (classes) of audio nodes allowed
struct Audio_worklet_thread {
    std::string name;
    Awnode *(*constructor)();
} audio_worklet_threads[MAX_THREADS];
int thread_index = 0;

void register_audio_worklet_class(std::string name, Awnode *(*constructor)())
{
    if (thread_index >= MAX_THREADS) {
        return;  // too many different node types
    }
    audio_worklet_threads[thread_index].name = name;
    audio_worklet_threads[thread_index].constructor = constructor;
    thread_index++;
}


int find_awnode_class(std::string name)
{
    for (int i = 0; i < thread_index; i++) {
        if (name.compare(audio_worklet_threads[i].name) == 0) {
            return i;
        }
    }
    return -1;
}


// a global where we save the context for use when we create an audio node:
static EMSCRIPTEN_WEBAUDIO_T gl_audio_context = NULL;

// This function will be called for every fixed 128 samples of audio to be
// processed.  It runs the process method of the Awnode subclass addressed
// by userData and deletes the object when processing is terminated by an
// EM_FALSE return value from process().
EM_BOOL awprocess(int numInputs, const AudioSampleFrame *inputs,
                  int numOutputs, AudioSampleFrame *outputs, int numParams,
                  const AudioParamFrame *params, void *userData)
{
    char msg_buffer[64];  // a receiving area for any message. NOTE that we
            // assume here that 64 > *any* message for any node. This supports
            // up to 16 32-bit fields or 8 64-bit fields.
    Awnode *awn = (Awnode *) userData;  // coerce data to object

    // process all incoming messages:
    while (Pm_Dequeue(awn->queue, msg_buffer)) {
        awn->message_handler(msg_buffer);
    }

    // run the sample processing method:
    if (!awn->process(numInputs, inputs, numOutputs, outputs,
                      numParams, params)) {
        delete awn;
        assert(false);  // NOTE: node deletion has not been tested and
        // is not really complete. We should at least remove awn from
        // audio_nodes[] so there is no dangling pointer there --
        // maybe the table index should be in every Awnode so it would
        // be easy to find and remove it from the table.
        //
        // Maybe there should also be a free list of open slots in
        // audio_nodes; otherwise, you will run out of slots after
        // creating MAX_NODES audio nodes.
        //
        // This is correct: return false to tell Web Audio that node
        // has ended:
        return EM_FALSE;
    }
    return EM_TRUE;
}


// generic function to deliver a message to an audio node. id is the node
// identifier, class_id is the node's subclass identifier (used to ensure
// we do not send a message for one node class to a node of a different
// class), and msg is a node-class-specific message of the appropriate
// length (the expected number of bytes is encoded into the node's queue,
// which is another reason we want to make sure the class_id is correct).
void send_message_to_node(int id, int class_id, char *msg)
{
    assert(0 <= id);
    assert(id < MAX_NODES);
    assert(msg);
    Awnode *awn = audio_nodes[id];
    assert(awn);
    assert(awn->class_id == class_id);
    assert(awn->queue);
    Pm_Enqueue(awn->queue, msg);
}



EM_JS(void, call_worklet_callback, (EMSCRIPTEN_WEBAUDIO_T audioContext,
                                    EM_BOOL success), {
    audioContext = emscriptenGetAudioObject(audioContext);
    window.emgl_audio_context = audioContext;
    window.emgl_audio_worklet_callback(success ? 0 : 1);
});


EM_JS(void, return_audio_node, (EMSCRIPTEN_AUDIO_WORKLET_NODE_T node), {
    node = emscriptenGetAudioObject(node);
    // console.log("return_audio_node", node);
    window.emgl_audio_node = node;
});
      

// This callback will fire after the Audio Worklet Processor has finished being
// added to the Worklet global scope.
void AudioWorkletProcessorCreated(EMSCRIPTEN_WEBAUDIO_T audioContext,
                                  EM_BOOL success, void *userData)
{
    call_worklet_callback(audioContext, success);
}


// This callback will fire when the Wasm Module has been shared to the
// AudioWorklet global scope, and is now ready to begin adding Audio
// Worklet Processors.
void WebAudioWorkletThreadInitialized(EMSCRIPTEN_WEBAUDIO_T audioContext,
                                      EM_BOOL success, void *userData)
{
    if (!success) return;
    WebAudioWorkletProcessorCreateOptions opts = { .name = "awnode" };
    emscripten_create_wasm_audio_worklet_processor_async(audioContext, &opts,
                                      AudioWorkletProcessorCreated, userData);
}


// Define a global stack space for the AudioWorkletGlobalScope. Note that all
// AudioWorkletProcessors and/or AudioWorkletNodes on the given Audio Context
// all share the same AudioWorkerGlobalScope, i.e. they all run on the same
// one audio thread (multiple nodes/processors do not each get their own
// thread). Hence one stack is enough.
uint8_t wasmAudioWorkletStack[4096];

bool main_has_run = false;


// create the audio worklet thread
int worklet_create()
{
    if (!main_has_run) {
        return -1;  // internal error; module creation failed or incomplete
    }

    if (!gl_audio_context) {
        // use default constructor options:
        gl_audio_context = emscripten_create_audio_context(0);

        emscripten_start_wasm_audio_worklet_thread_async(gl_audio_context,
                wasmAudioWorkletStack, sizeof(wasmAudioWorkletStack),
                WebAudioWorkletThreadInitialized, 0);
    }
    return 0;
}


// Create an audio node instance.
// returns < 0 on error; otherwise returns a node instance id >= 0.
// The id could be used to find the node's state structure.
int audio_node_create(std::string name) {
    // Specify the input and output node configurations for the Wasm Audio
    // Worklet. A simple setup with single mono output channel here,
    // and no inputs.
    int outputChannelCounts[1] = { 1 };

    EmscriptenAudioWorkletNodeCreateOptions options = {
            .numberOfInputs = 0,
            .numberOfOutputs = 1,
            .outputChannelCounts = outputChannelCounts
    };

    if (node_index >= MAX_NODES) return -2;  // too many nodes
    int my_index = find_awnode_class(name);
    if (my_index < 0) {
        return -2;  // invalid or unregistered name
    }

    Awnode *awn = (*(audio_worklet_threads[my_index].constructor))();
    if (!awn) {
        return -3;  // constructor failed
    }

    // Instantiate the named Audio Worklet Processor.
    EMSCRIPTEN_AUDIO_WORKLET_NODE_T wasmAudioWorklet =
            emscripten_create_wasm_audio_worklet_node(gl_audio_context,
                                   "awnode", &options, &awprocess, awn);

    // round-about way to return a value -- if we return it directly, an
    // integer is returned through the EMSCRIPTEN_BINDINGS method even if
    // we declare this function to be of type EMSCRIPTEN_AUDIO_WORKLET_NODE_T
    return_audio_node(wasmAudioWorklet);

    audio_nodes[node_index] = awn;

    if (wasmAudioWorklet == NULL) {
        return -1;  // returns negative number on error
    } else {
        return node_index++;  // return this node's id
    }
}


EMSCRIPTEN_BINDINGS(awutil_bindings) {
    emscripten::function("worklet_create", &worklet_create);
    emscripten::function("audio_node_create", &audio_node_create);
}
