/* noise.cpp -- an audio worklet kernel for "tone"
 *
 * Roger B. Dannenberg
 * Oct 2023
 */

#include "awutil.h"
#include "noise.h"

// int32_t is the "native" element for Pm_Queue, so it is preferred:
enum Opcode : int32_t {SET_GAIN = 0};

struct Noise_msg {
    Opcode opcode;
    float val;
};


Noise::Noise()
{
    // Note: subclass MUST initialize class_id for consistency checks:
    class_id = 'nois';
    // creates space for 8 pending messages -- be careful and see pmutil.h
    // for details of overflow which is not handled in this example.
    queue = Pm_QueueCreate(8, (int) sizeof(Noise_msg));
    stop = false;
    stopped = false;
    gain = 0.1;
}


Noise::~Noise()
{
    // NOTE: queue is destroyed by ~Awnode, so leave it alone here.
    return;
}


Awnode *noise_constructor()
{
    return new Noise();
}


EM_BOOL Noise::process(int numInputs, const AudioSampleFrame *inputs,
                          int numOutputs, AudioSampleFrame *outputs,
                          int numParams, const AudioParamFrame *params)
{
    Noise_msg msg;
    // Produce tone in output channel 0.
    for (int j = 0; j < 128; ++j) {
        outputs[0].data[j] = (rand() / ((float) RAND_MAX)) * gain;
    }
    // copy all k samples from output 0 channel 0 to all outputs i channels j
    mono_to_all(numOutputs, outputs);
    // We generated audio and want to keep this processor going.
    // Return EM_FALSE here to shut down.
    stopped = stop;  // let world know this state and node is free now
    return !stop;
}


void Noise::message_handler(char *msg)
{
    Noise_msg *stm = (Noise_msg *) msg;

    if (stm->opcode == SET_GAIN) {
        gain = stm->val;
    }
}


void noise_set_gain(int id, float g)
{
    Noise_msg msg;
    msg.opcode = SET_GAIN;
    msg.val = g;
    
    send_message_to_node(id, 'nois', (char *) &msg);
}


EMSCRIPTEN_BINDINGS(noise_bindings) {
    emscripten::function("worklet_create", &worklet_create);
    emscripten::function("audio_node_create", &audio_node_create);
    emscripten::function("noise_set_gain", &noise_set_gain);
}
