/* sinetone.cpp -- an audio worklet kernel for "tone"
 *
 * Roger B. Dannenberg
 * Oct 2023
 */

#include "awutil.h"
#include "sinetone.h"

// int32_t is the "native" element for Pm_Queue, so it is preferred:
enum Opcode : int32_t {SET_GAIN = 0, SET_FREQ = 1};

struct Sinetone_msg {
    Opcode opcode;
    float val;
};


Sinetone::Sinetone()
{
    // Note: subclass MUST initialize class_id for consistency checks:
    class_id = 'sntn';
    // creates space for 8 pending messages -- be careful and see pmutil.h
    // for details of overflow which is not handled in this example.
    queue = Pm_QueueCreate(8, (int) sizeof(Sinetone_msg));
    stop = false;
    stopped = false;
    gain = 0.1;
    phase = 0.0;
    phase_inc = 880 * M_PI * 2 / 48000;
}


Sinetone::~Sinetone()
{
    // NOTE: queue is destroyed by ~Awnode, so leave it alone here.
    return;
}


Awnode *sinetone_constructor()
{
    return new Sinetone();
}


EM_BOOL Sinetone::process(int numInputs, const AudioSampleFrame *inputs,
                          int numOutputs, AudioSampleFrame *outputs,
                          int numParams, const AudioParamFrame *params)
{
    Sinetone_msg msg;
    // Produce tone in output channel 0.
    for (int j = 0; j < 128; ++j) {
        outputs[0].data[j] = sin(phase) * gain;
        phase = fmod(phase + phase_inc, M_PI * 2);
    }
    // copy all k samples from output 0 channel 0 to all outputs i channels j
    mono_to_all(numOutputs, outputs);
    // We generated audio and want to keep this processor going.
    // Return EM_FALSE here to shut down.
    stopped = stop;  // let world know this state and node is free now
    return !stop;
}


void Sinetone::message_handler(char *msg)
{
    Sinetone_msg *stm = (Sinetone_msg *) msg;

    if (stm->opcode == SET_GAIN) {
        gain = stm->val;
    } else if (stm->opcode == SET_FREQ) {
        phase_inc = stm->val * M_PI * 2 / 48000;
    }
}


void sinetone_set_gain(int id, float g)
{
    Sinetone_msg msg;
    msg.opcode = SET_GAIN;
    msg.val = g;
    
    send_message_to_node(id, 'sntn', (char *) &msg);
}


void sinetone_set_freq(int id, float hz)
{
    Sinetone_msg msg;
    msg.opcode = SET_FREQ;
    msg.val = hz;
    
    send_message_to_node(id, 'sntn', (char *) &msg);
}


EMSCRIPTEN_BINDINGS(sinetone_bindings) {
    emscripten::function("sinetone_set_gain", &sinetone_set_gain);
    emscripten::function("sinetone_set_freq", &sinetone_set_freq);
}
